/*

	This is the source code of

   	M A I T R E Y A
    ===============

	Open source platform for Vedic and western astrology

  File           Calculator.cpp
  Release        4.1
  Author         Martin Pettau
  Copyright (C)  2003-2006 by the author

  Released under the Artistic License as published by the
  Free Software Foundation, read the file 'COPYING' for more information.

*/

#ifdef __GNUG__
	#pragma implementation "Calculator.h"
#endif

#include "Calculator.h"

#include<wx/log.h>
#include <wx/string.h>

#ifndef TRUE
#define TRUE 1
#endif
#define TRUE 1

#ifdef __WXMSW__
#undef TRUE
#endif
#ifndef _cplusplus
extern "C" {
#endif
#include "swephexp.h"
#include "swepcalc.h"
#ifndef _cplusplus
}
#endif

#include "constants.h"
#include "func.h"
#include "Horoscope.h"
#include "Lang.h"
#include "Conf.h"
#include "Session.h"
#include "Lang.h"

extern Config *config;

/*****************************************************
**
**   EclipseData   ---   Constructor 
**
******************************************************/
EclipseData::EclipseData()
{
	jd = sunpos = moonpos = 0;
	planet = type = 0;
	description = wxT( "" );
}

/*****************************************************
**
**   EclipseData   ---   Constructor 
**
******************************************************/
EclipseData::EclipseData( const EclipseData &d )
{
	jd = d.jd;
	sunpos = d.sunpos;
	moonpos = d.moonpos;
	planet = d.planet;
	type = d.type;
	description = d.description;
}

/*****************************************************
**
**   CalculatorSwe   ---   Constructor 
**
******************************************************/
CalculatorSwe::CalculatorSwe()
{
	pedantic = false;
	//pedantic = true;
	init();
}

/*****************************************************
**
**   CalculatorSwe   ---   Destructor 
**
******************************************************/
CalculatorSwe::~CalculatorSwe()
{
	swe_close();
}

/*****************************************************
**
**   CalculatorSwe   ---   init 
**
******************************************************/
void CalculatorSwe::init()
{
	char ww[4096];
	strcpy( ww, wxConvertWX2MB( config->sePath.c_str() ));
	if ( pedantic && ww ) printf( "Ephem path: %s\n", ww );
	swe_set_ephe_path( (char*)ww );
}

struct AyaInit {double t0, ayan_t0;};
static const struct AyaInit Ayanamsa[NB_AYANAMSAS] = {
	{2415020.0, 360 - 337.53953},   /* 1: Lahiri (Robert Hand) */
	{2415020.0, 360 - 338.98556},   /* 3: Raman (Robert Hand) */
	{2415020.0, 360 - 337.636111}  /* 5: Krishnamurti (Robert Hand) */
};

/*****************************************************
**
**   CalculatorSwe   ---   calcAyanamsa 
**
******************************************************/
double CalculatorSwe::calcAyanamsa( const double &jd, const int &type )
{
	double t, t0, ayan_t0;
	assert( type >= 0 && type <= NB_AYANAMSAS+1 );
	if ( type == 0 ) return 0;

	if ( type <= NB_AYANAMSAS && type > 0 )
	{
		t0 = Ayanamsa[type-1].t0;
		ayan_t0 = Ayanamsa[type-1].ayan_t0;
	}
	else
	{
		t0 = config->custom_t0;
		ayan_t0 = config->custom_ayan_t0;
	}
	if ( config->custom_aya_constant )
	{
		t = jd - t0;
		double years = t/365.25;
		double portion = years/config->custom_aya_period;
		double aya = portion * 360;
		//return red_deg( config->custom_ayan_t0 + 360 * ( jd - config->custom_t0 ) / ( config->custom_aya_period * 365.25 ));
		return red_deg( aya );
	}
	else
	{
		swe_set_sid_mode( 255, t0, ayan_t0 );
		return swe_get_ayanamsa( calcJd( jd ));
	}
}

/*****************************************************
**
**   CalculatorSwe   ---   calcSiderealTime 
**
******************************************************/
double CalculatorSwe::calcSiderealTime( const double &jd , const double &longitude )
{
	return swe_sidtime( jd + longitude / 360 );
}

/*****************************************************
**
**   CalculatorSwe   ---   calcEps 
**
******************************************************/
void CalculatorSwe::calcEps( double &eps, const double &jd )
{
	char err[256];
	double x[6];
				
	// x[0] enthaelt eps
	if ( swe_calc( calcJd( jd ), SE_ECL_NUT, 0, x, err) == -1 )
	{
		assert( false );
	}
	if ( pedantic && err && strlen( err ) > 0 ) printf( "Swe: %s\n", err );
	eps = x[0];
}

/*****************************************************
**
**   CalculatorSwe   ---   calcSunRiseSunSet 
**
******************************************************/
bool CalculatorSwe::calcSunRiseSunSet( DataSet *d, double &sunrise, double &sunset )
{
	char err[256];
	bool daybirth = true, beforemidnight = true;
	double r[6];
	double cusp[13], ascmc[10];
	double rsmi[3], tret[3];
	double startjd1, startjd2;
	int flag = 0;

	if ( ! config->sunrise_def ) flag = SE_BIT_DISC_CENTER;
	if ( ! config->sunrise_refrac ) flag |= SE_BIT_NO_REFRACTION;
	double jd = calcJd( d->getJD() );

	swe_calc( jd, OSUN, 0, r,err );
	if ( pedantic && err && strlen( err ) > 0 ) printf( "Swe: %s\n", err );
	swe_houses( jd, d->getLocation()->getLatitude(), d->getLocation()->getLongitude(), 'E', cusp, ascmc );

	double diff_ascsun = red_deg( ascmc[0] -  r[0] );  // Sun and AC
	if ( diff_ascsun > 180 ) daybirth = false;

	double diff_icsun = red_deg( ascmc[1] + 180 - r[0] );  // Sun and IC
	if ( diff_icsun < 180 ) beforemidnight = false;

	if ( daybirth )
	{
		startjd1 = jd-1;
		startjd2 = jd;
	}
	else
	{
		if ( beforemidnight )
		{
			startjd1 = jd - 1;
			startjd2 = jd - 1;
		}
		else
		{
			startjd1 = jd;
			startjd2 = jd;
		}
	}
	rsmi[0] = d->getLocation()->getLongitude();
	rsmi[1] = d->getLocation()->getLatitude();
	rsmi[2] = 0;
	swe_rise_trans( startjd1, SE_SUN, 0, 0, SE_CALC_RISE|flag, rsmi, 0, 0, tret, err );
	if ( pedantic && err && strlen( err ) > 0 ) printf( "Swe: %s\n", err );
	sunrise = tret[0];
	swe_rise_trans( startjd2, SE_SUN, 0, 0, SE_CALC_SET|flag, rsmi, 0, 0, tret, err );
	if ( pedantic && err && strlen( err ) > 0 ) printf( "Swe: %s\n", err );
	sunset = tret[0];
	return daybirth;
}

/*****************************************************
**
**   CalculatorSwe   ---   CalcDateForAscendantPos 
**
******************************************************/
double CalculatorSwe::CalcDateForAscendantPos( DataSet *d, const double &dest )
{
	double diff, alen, dummy, speed;
	int count = 0;
	DataSet dataset( *d );
	Location *loc = dataset.getLocation();
	double lat = loc->getLatitude();
	double lon = loc->getLongitude();
	speed = 720;
	do
	{
		//calcPositionSpeed( &dataset, 1, alen, dummy, mspeed, true, true );
		alen = calcAscendantAya( dataset.getJD(), lat, lon, true );

		diff = alen - dest;
		if ( diff > 200 ) diff -= 360;
		if ( diff < -200 ) diff += 360;
		printf( "Alen %f dest %f diff %f\n", alen, dest, diff );
		dataset.setDate( dataset.getJD() - diff / speed );
	}
	while (( fabs( diff ) > 0.0001 ) && ( count++ < 40 ));
	printf( "Fertig: %d\n", count );

	if ( count > 39 ) printf("Too many iterations for calculation of ascecndant length\n" );
	return dataset.getJD();
}

/*****************************************************
**
**   CalculatorSwe   ---   calcAscendantPortion 
**
******************************************************/
double CalculatorSwe::calcAscendantPortion( DataSet *d, const double &alen )
{
	int nakshatra;
	double nak_len, jd1, jd2, len1, len2;

	nakshatra = (int)( alen / NAKSHATRA_LEN );
	nak_len = alen - nakshatra * NAKSHATRA_LEN;
	printf( "nak %d len %f\n", nakshatra, nak_len );
	if ( config->iNakshatraPortionMode ) // according to entry time
	//if ( 0 )
	{
		printf( "METHOD1\n" );
		len1 = nakshatra * NAKSHATRA_LEN;
		len2 = ( nakshatra + 1 ) * NAKSHATRA_LEN;
		jd1 = CalcDateForAscendantPos( d, len1 );
		jd2 = CalcDateForAscendantPos( d, len2 );
		return ( d->getJD() - jd1 ) / ( jd2 - jd1 );
	}
	else // according to length
	{
		printf( "METHOD1\n" ); // mars 1969-9.10
		return nak_len / NAKSHATRA_LEN;
	}
}

/*****************************************************
**
**   CalculatorSwe   ---   calcNakshatraPortion 
**
******************************************************/
double CalculatorSwe::calcNakshatraPortion( DataSet *d, const double &mlen, const bool usepada )
{
	int nakshatra, pada;
	double nak_len, p_len, len1, len2, jd1, jd2;

	nakshatra = (int)( mlen / NAKSHATRA_LEN );
	nak_len = mlen - nakshatra * NAKSHATRA_LEN;
	pada = (int)(nak_len / PADA_LEN );
	p_len = mlen - ( nakshatra * NAKSHATRA_LEN + pada * PADA_LEN );
	if ( usepada )
	{
		len1 = nakshatra * NAKSHATRA_LEN + pada * PADA_LEN;
		len2 = nakshatra * NAKSHATRA_LEN + ( pada + 1 ) * PADA_LEN;
	}
	else
	{
		len1 = nakshatra * NAKSHATRA_LEN;
		len2 = ( nakshatra + 1 ) * NAKSHATRA_LEN;
	}

	if ( config->iNakshatraPortionMode ) // according to entry time
	{
		jd1 = CalcDateForMoonPos( d, len1 );
		jd2 = CalcDateForMoonPos( d, len2 );
		return ( d->getJD() - jd1 ) / ( jd2 - jd1 );
	}
	else // according to length
	{
		if ( ! usepada ) return nak_len / NAKSHATRA_LEN;
		else return p_len / PADA_LEN;
	}
}

/*****************************************************
**
**   CalculatorSwe   ---   calcPositionSpeed 
**
******************************************************/
void CalculatorSwe::calcPositionSpeed( DataSet *d, const int &planet, double &slen, double &slat, double &sspeed,
		const bool aya, const bool indian )
{
	double r[6];
	char err[256];
	long iflag = SEFLG_SPEED;

	if ( config->useTruePositions )
	{
		iflag |= SEFLG_TRUEPOS;
	}
	if ( config->useTopocentric )
	{
		swe_set_topo( d->getLocation()->getLongitude(), d->getLocation()->getLatitude(), 0 );
		iflag |= SEFLG_TOPOCTR;
	}

	swe_calc( calcJd( d->getJD() ), mapOToS( planet ), iflag, r, err );
	if ( pedantic && err && strlen( err ) > 0 ) printf( "Swe: %s\n", err );
	slen = red_deg( r[0] - ( aya ? calcAyanamsa( d->getJD(), ( indian ? config->iAyanamsa : config->wAyanamsa )) : 0 ));
	slat = r[1];
	sspeed = r[3];
}

/*****************************************************
**
**   CalculatorSwe   ---   calcPosition 
**
******************************************************/
void CalculatorSwe::calcPosition( DataSet *d, const int &planet, double &slen, double &slat, 
		const bool aya, const bool indian )
{
	double r[6];
	char err[256];
	long iflag = 0;

	if ( config->useTruePositions )
	{
		iflag |= SEFLG_TRUEPOS;
	}
	if ( config->useTopocentric )
	{
		swe_set_topo( d->getLocation()->getLongitude(), d->getLocation()->getLatitude(), 0 );
		iflag |= SEFLG_TOPOCTR;
	}

	swe_calc( calcJd( d->getJD() ), mapOToS( planet ), iflag, r, err );
	if ( pedantic && err && strlen( err ) > 0 ) printf( "Swe: %s\n", err );
	slen = red_deg( r[0] - ( aya ? calcAyanamsa( d->getJD(), ( indian ? config->iAyanamsa : config->wAyanamsa )) : 0 ));
	slat = r[1];
}

/*****************************************************
**
**   CalculatorSwe   ---   calcPosSpeedAya 
**
******************************************************/
void CalculatorSwe::calcPosSpeedAya( const double &jd, const int &planet, double &slen, double &slat,
	double &sspeed, bool indian )
{
	double ayanamsa = 0;
	double r[6];
	char err[256];

	indian ? ayanamsa = calcAyanamsa( jd, config->iAyanamsa ) : calcAyanamsa( jd, config->wAyanamsa );

	swe_calc( calcJd( jd ), mapOToS( planet ), SEFLG_SPEED, r, err );
	if ( pedantic && err && strlen( err ) > 0 ) printf( "Swe: %s\n", err );
	slen = red_deg( r[0] - ayanamsa );
	slat = r[1];
	sspeed = r[3];
}

/*****************************************************
**
**   CalculatorSwe   ---   calcPosSpeed 
**
******************************************************/
void CalculatorSwe::calcPosSpeed( const double &jd, const int &planet, double &slen, double &slat, double &sspeed )
{
	double r[6];
	char err[256];

	swe_calc( calcJd( jd ), mapOToS( planet ), SEFLG_SPEED, r, err );
	if ( pedantic && err && strlen( err ) > 0 ) printf( "Swe: %s\n", err );
	slen = r[0];
	slat = r[1];
	sspeed = r[3];
}

/*****************************************************
**
**   CalculatorSwe   ---   calcPosAya 
**
******************************************************/
double CalculatorSwe::calcPosAya( const double &jd, const int &planet, bool indian )
{
	double ayanamsa = 0;
	double r[6];
	char err[256];

	indian ? ayanamsa = calcAyanamsa( jd, config->iAyanamsa ) : calcAyanamsa( jd, config->wAyanamsa );

	swe_calc( calcJd( jd ), mapOToS( planet ), 0, r, err );
	if ( pedantic && err && strlen( err ) > 0 ) printf( "Swe: %s\n", err );
	return red_deg( r[0] - ayanamsa );
}

/*****************************************************
**
**   CalculatorSwe   ---   calcPos 
**
******************************************************/
double CalculatorSwe::calcPos( const double &jd, const int &planet )
{
	double r[6];
	char err[256];

	swe_calc( calcJd( jd ), mapOToS( planet ), 0, r, err );
	if ( pedantic && err && strlen( err ) > 0 ) printf( "Swe: %s\n", err );
	return r[0];
}

/*****************************************************
**
**   CalculatorSwe   ---   calcHouses 
**
******************************************************/
void CalculatorSwe::calcHouses( const double &jd, const double &lon, const double &lat, const int &hsys, double *r )
{
	double cusp[13], ascmc[10];
	int c = 'P', i;
	switch( hsys )
	{
		case HOUSE_SYSTEM_NONE:
		case HOUSE_SYSTEM_PLACIDUS:
			c = 'P';
		break;
		case HOUSE_SYSTEM_KOCH:
			c = 'K';
		break;
		case HOUSE_SYSTEM_REGIOMONTANUS:
			c = 'R';
		break;
		case HOUSE_SYSTEM_CAMPANUS:
			c = 'C';
		break;
		case HOUSE_SYSTEM_PORPHYRY:
			c = 'O';
		break;
		case HOUSE_SYSTEM_EQUAL:
			c = 'E';
		break;
		case HOUSE_SYSTEM_VEHLOW:
			c = 'V';
		break;
		case HOUSE_SYSTEM_AXIAL:
			c = 'X';
		break;
		case HOUSE_SYSTEM_AZIMUTAL:
			c = 'H';
		break;
		case HOUSE_SYSTEM_POLICH:
			c = 'T';
		break;
		case HOUSE_SYSTEM_ALCABITUS:
			c = 'B';
		break;
		/*
		case HOUSE_SYSTEM_GAUQUELIN:
			c = 'G';
		break;
		*/
		case HOUSE_SYSTEM_MORINUS:
			c = 'M';
		break;
	}
	swe_houses( jd, lat, lon, c, cusp, ascmc );
	for( i = 0; i < 12; i++ ) r[i] = cusp[1+i];
}

/*****************************************************
**
**   CalculatorSwe   ---   CalcDateForMoonPos 
**
******************************************************/
double CalculatorSwe::CalcDateForMoonPos( DataSet *d, const double &dest )
{
	double diff, mlen, dummy, mspeed;
	int count = 0;
	DataSet dataset( *d );
	do
	{
		calcPositionSpeed( &dataset, 1, mlen, dummy, mspeed, true, true );

		diff = mlen - dest;
		if ( diff > 200 ) diff -= 360;
		if ( diff < -200 ) diff += 360;
		dataset.setDate( dataset.getJD() - diff / mspeed );
	}
	while (( fabs( diff ) > 0.0000001 ) && ( count++ < 20 ));

	if ( count > 19 ) printf("Too many iterations for calculation of moon length\n" );
	return dataset.getJD();
}

/*****************************************************
**
**   CalculatorSwe   ---   calc_jd 
**
******************************************************/
double CalculatorSwe::calc_jd( const int &year, const int &month, const int &day, const double &time )
{
	return swe_julday( year, month, day, time, 1 );
}

/*****************************************************
**
**   CalculatorSwe   ---   calcTajakaJd 
**
******************************************************/
double CalculatorSwe::calcTajakaJd( const DataSet *dataset, const int &tyear, const bool &vedic )
{
	double diff, slen, dummy, speed, birthlen;
	int count = 0;
	int dummy2, dummy3, year;
	Formatter::get()->getDateIntsFromJD( dataset->getJD(), dummy2, dummy3, year );

	DataSet d( *dataset );
	calcPosition( &d, 0, birthlen, dummy, true, vedic );
	d.setDate( dataset->getJD() + ( tyear - year ) * 365.25 );
	do
	{
		calcPositionSpeed( &d, 0, slen, dummy, speed, true, vedic );

		diff = slen - birthlen;
		if ( diff > 200 ) diff -= 360;
		if ( diff < -200 ) diff += 360;
		d.setDate( d.getJD() - diff / speed );
	}
	while (( fabs( diff ) > 0.0000001 ) && ( count++ < 20 ));

	if ( count > 19 ) printf( "Too many iterations for calculation of Sun length\n" );
	return d.getJD();
}

/***************************************************************
**
** CalculatorSwe   ----  getDateIntsFromJD
**
***************************************************************/
bool CalculatorSwe:: getDateIntsFromJD( const double &jd, int &year, int &month, int &day, double &ut )
{
	swe_revjul ( jd, 1, &year, &month, &day, &ut );
	return true;
}

/*****************************************************
**
**   CalculatorSwe   ---   calcAscendant 
**
******************************************************/
double CalculatorSwe::calcAscendant( const double &jd, const double &lat, const double &lon )
{
	double cusp[13], ascmc[10];
	swe_houses( jd, lat, lon, 'P', cusp, ascmc );

	return ascmc[0];
}

/*****************************************************
**
**   CalculatorSwe   ---   calcAscendantAya 
**
******************************************************/
double CalculatorSwe::calcAscendantAya( const double &jd, const double &lat, const double &lon, bool indian )
{
	double ayanamsa = 0;
	double cusp[13], ascmc[10];
	swe_houses( jd, lat, lon, 'P', cusp, ascmc );

	ayanamsa = calcAyanamsa( calcJd( jd ), ( indian ? config->iAyanamsa : config->wAyanamsa ));
	return red_deg( ascmc[0] - ayanamsa );
}

/*****************************************************
**
**   CalculatorSwe   ---   calcAscendantByArmc 
**
******************************************************/
double CalculatorSwe::calcAscendantByArmc( const double &armc, const double &eps, const double &lat, double *r )
{
	int i;
	double cusp[13], ascmc[10];
	swe_houses_armc( armc, lat, eps, 'P', cusp, ascmc );
	if ( r != 0 )
	{
		for( i = 0; i < 12; i++ )
		{
			r[i] = cusp[1+i];
		}
	}
	return ascmc[0];
}

/*****************************************************
**
**   CalculatorSwe   ---   calcMC 
**
******************************************************/
double CalculatorSwe::calcMC( const double &jd, const double &lat, const double &lon )
{
	double cusp[13], ascmc[10];
	swe_houses( jd, lat, lon, 'P', cusp, ascmc );

	return ascmc[1];
}

/*****************************************************
**
**   CalculatorSwe   ---   calcMCAya 
**
******************************************************/
double CalculatorSwe::calcMCAya( const double &jd, const double &lat, const double &lon, bool indian )
{
	double cusp[13], ascmc[10];
	swe_houses( jd, lat, lon, 'P', cusp, ascmc );

	return red_deg( ascmc[1] - calcAyanamsa( jd, ( indian ? config->iAyanamsa : config->wAyanamsa )  ));
}

/*****************************************************
**
**   CalculatorSwe   ---   calcNextEclipse 
**
******************************************************/
EclipseData CalculatorSwe::calcNextEclipse( const double &jd, const int &planet, const int &type, const bool &vedic )
{
	EclipseData data;
	double tret[10];
	char err[256];
	int ret;

	if ( planet == OSUN ) ret = swe_sol_eclipse_when_glob( calcJd( jd ), 0, 0, tret, false, err );
	else ret = swe_lun_eclipse_when( calcJd( jd ), 0, 0, tret, false, err );
	if ( pedantic && err && strlen( err ) > 0 ) printf( "Swe: %s\n", err );
	data.jd = tret[0];
	data.type = ret;
	data.planet = planet;
	data.sunpos = calcPosAya( data.jd, OSUN, vedic );
	data.moonpos = calcPosAya( data.jd, OMOON, vedic );
	if ( ret & SE_ECL_TOTAL ) data.description = _( "Total" );
	else if ( ret & SE_ECL_ANNULAR ) data.description = _( "Annular" );
	else if ( ret & SE_ECL_PARTIAL ) data.description = _( "Partial" );
	else if ( ret & SE_ECL_PENUMBRAL ) data.description = _( "Penumbral" );
	else if ( ret & SE_ECL_ANNULAR_TOTAL ) data.description = _( "Annular Total" );
	else data.description = _( "unknown" );

	return data;
}

/*****************************************************
**
**   CalculatorSwe   ---   calcMcFromAsc 
**
******************************************************/
double CalculatorSwe::calcMcFromAsc( const double &ascpos, const double &jd, const double &lon, const double &lat )
{
	double pos[3], xin[2], xaz[3];
	pos[0] = lon;
	pos[1] = lat;
	pos[2] = 0;

	xin[0] = ascpos;
	xin[1] = 0;
	//xin[0] = 0;

	swe_azalt( jd, 	SE_ECL2HOR, pos, 0, 10, xin, xaz );
	//printf( "Result: %f %f %f\n", xaz[0], xaz[1], xaz[2] );
	return 0;
}

/*****************************************************
**
**   CalculatorSwe   ---   calcPlanetaryEvent 
**
******************************************************/
double CalculatorSwe::calcPlanetaryEvent( const DataSet *d, const double &targetlen, const int &planet,
	const bool &vedic )
{
	double diff, len, dummy, speed;
	DataSet dataset( *d );

	int count = 0;
	int pp = planet;

	do
	{
		// map both rahu and ketu to the same node (depending on conf for inode)
		if ( vedic && ( planet == OMEANNODE || planet == OTRUENODE ))
			pp = ( config->iLunarNodeMode == LUNAR_NODE_TRUE ? OTRUENODE : OMEANNODE );

		calcPositionSpeed( &dataset, pp, len, dummy, speed, true, vedic );

		if ( vedic && planet == OTRUENODE ) len = red_deg( len + 180 ); // ketu gets len of rahu + 180

		diff = len - targetlen;
		if ( diff > 200 ) diff -= 360;
		if ( diff < -200 ) diff += 360;
		dataset.setDate( dataset.getJD() - diff / speed );
	}
	while (( fabs( diff ) > 0.0000001 ) && ( count++ < 20 ));

	if ( count > 19 )
	{
		printf( "Too many iterations for calculation of planetary length\n" );
	}
	return dataset.getJD();
}

/*****************************************************
**
**   CalculatorSwe   ---   calcSunMoonEvent
**
******************************************************/
double CalculatorSwe::calcSunMoonEvent( DataSet *d, const double &angle, double &slen, double &mlen )
{
	double mspeed, sspeed, dummy, a, diff;
	int count = 0;
	const double targettol = .000001;
	do
	{
		calcPositionSpeed( d, OSUN, slen, dummy, sspeed, true, false );
		calcPositionSpeed( d, OMOON, mlen, dummy, mspeed, true, false );
		a = red_deg( mlen - slen );
		diff = angle - a;
		if ( diff > 200 ) diff -= 360;
		if ( diff < -200 ) diff += 360;
		//printf( "Calculator: target %f count %d %.15f diff is %.15f\n", angle, count, a, diff );
		d->setDate( d->getJD() + diff / ( mspeed - sspeed ));

		count++;
	}
	while ( fabs( diff ) > targettol && count < 20 );

	//printf( "\n" );
	return d->getJD();
}

/*****************************************************
**
**   CalculatorSwe   ---   calcJd 
**
******************************************************/
double CalculatorSwe::calcJd( const double &jd )
{
	return( config->useEphemerisTime ? jd : jd + swe_deltat( jd ));
}

/*****************************************************
**
**   CalculatorSwe   ---   mapOToS 
**
******************************************************/
int CalculatorSwe::mapOToS( const int &i )
{
	switch( i )
	{
		case OSUN      : return SE_SUN;       break;
		case OMOON     : return SE_MOON;      break;
		case OMERCURY  : return SE_MERCURY;   break;
		case OVENUS    : return SE_VENUS;     break;
		case OMARS     : return SE_MARS;      break;
		case OJUPITER  : return SE_JUPITER;   break;
		case OSATURN   : return SE_SATURN;    break;
		case OURANUS   : return SE_URANUS;    break;
		case ONEPTUNE  : return SE_NEPTUNE;   break;
		case OPLUTO    : return SE_PLUTO;     break;
		case OMEANNODE : return SE_MEAN_NODE; break;
		case OTRUENODE : return SE_TRUE_NODE; break;
		case OCUPIDO   : return SE_CUPIDO;    break;
		case OHADES    : return SE_HADES;     break;
		case OZEUS     : return SE_ZEUS;      break;
		case OKRONOS   : return SE_KRONOS;    break;
		case OAPOLLON  : return SE_APOLLON;   break;
		case OADMETOS  : return SE_ADMETOS;   break;
		case OVULKANUS : return SE_VULKANUS;  break;
		case OPOSEIDON : return SE_POSEIDON;  break;
		default        : assert( 0 );         break;
	}
	return -1;
}

