/*

	This is the source code of

   	M A I T R E Y A
    ===============

	Open source platform for Vedic and western astrology

  File           Ephemeris.cpp
  Release        4.1
  Author         Martin Pettau
  Copyright (C)  2003-2006 by the author

  Released under the Artistic License as published by the
  Free Software Foundation, read the file 'COPYING' for more information.

*/

#ifdef __GNUG__
	#pragma implementation "Ephemeris.h"
#endif

#include "Ephemeris.h"

#include<wx/log.h>

#include "func.h"
#include "Calculator.h"
#include "Conf.h"
#include "DataSet.h"
#include "Lang.h"
#include "Session.h"
#include "Writer.h"

extern Config *config;

/*****************************************************
**
**   CLASS IngressEventSort  
**
******************************************************/
class IngressEventSort
{
public:
  bool operator()( const IngressEvent &e1, const IngressEvent &e2 )
  {
    return( e1.jd < e2.jd );
  }
};

/*****************************************************
**
**   CLASS LunarEventSort  
**
******************************************************/
class LunarEventSort
{
public:
  bool operator()( const LunarEvent &e1, const LunarEvent &e2 )
  {
    return( e1.jd < e2.jd );
  }
};

/*****************************************************
**
**   EphemExpert   ---   Constructor 
**
******************************************************/
EphemExpert::EphemExpert()
{
	d = new DataSet();
	d->setLocation( config->defaultLocation );
	this->preferVedic = config->preferVedic;
	init();
}

/*****************************************************
**
**   EphemExpert   ---   Destructor 
**
******************************************************/
EphemExpert::~EphemExpert()
{
	delete d;
}

/*****************************************************
**
**   EphemExpert   ---   setPreferVedic 
**
******************************************************/
void EphemExpert::setPreferVedic( const bool &p )
{
	preferVedic = p;
	init();
}

/*****************************************************
**
**   EphemExpert   ---   init 
**
******************************************************/
void EphemExpert::init()
{
	int i, j;
	clen = cingress = ckp = cdetails = clunar = false;
	mydasa = -1;

	// reset variables
	for( i = 0; i <= 31; i++ )
	{
		weekday[i] = tithi[i] = 0;
		jd[i] = st[i] =  sunrise[i] = sunset[i] = 0;
		for( j = 0; j <= OPOSEIDON; j++ )
		{
			len[j][i]   = speed[j][i]   = 0;
			rasi[j][i]  = nakshatra[j][i] = 0;
			retro[j][i] = false;
		}
	}
}

/*****************************************************
**
**   EphemExpert   ---   prepareMonth 
**
******************************************************/
void EphemExpert::prepareMonth( const int &m, const int &y, const bool &isLocaltime )
{
	int dummy;
	Formatter *formatter = Formatter::get();
	year = y;
	month = m;
	localtime = isLocaltime;
	tz = localtime ? config->defaultLocation.getTimeZone() + config->defaultLocation.getDST() : 0;
	assert( month >= 0 && month <= 12 );
	assert( year >= -2000 && year <= 4000 );

	clen = cingress = ckp = cdetails = clunar = false;

	nb_days = 30;
	if ( month == 0 ) formatter->getDateIntsFromJD( d->getJD(), dummy, month, year );

  if ( month == 1 || month == 3 || month == 5 || month == 7 || month == 8 || month == 10 || month == 12 ) nb_days = 31;
  if ( month == 2 ) // February
  {
    if ( year % 4 ) nb_days = 28; else nb_days = 29;
    if (! year % 100 ) nb_days = 28;
    if (! year % 400 ) nb_days = 29;
  }

	// set jds and week days
	for( int i = 0; i <= nb_days; i++ ) // calc JDs for this month and year
	{
		d->setDate( i+1, month, year, - tz );
		jd[i] = d->getJD();
    weekday[i] = ((int)(jd[i] + 2 )) % 7;
	}
}

/*****************************************************
**
**   EphemExpert   ---   calcLength 
**
******************************************************/
void EphemExpert::calcLength()
{
	int i, j, nodeindex;
	double l, s, dummy, nodelen;
	bool nodedir;
	Calculator *calculator = Session::get()->getCalculator();

	for( i = 0; i <= nb_days; i++ ) // calc length and retrogression
	{
		d->setDate( i+1, month, year, -tz );
		for( j = 0; j <= OPOSEIDON; j++ )
		{
			// make it faster and exclude some planets in vedic mode
			if ( preferVedic && ( j >= OCUPIDO || ( j > OSATURN && j < OMEANNODE ))) continue;

			calculator->calcPositionSpeed( d, j, l, dummy, s, true, preferVedic );
			len[j][i]   = l;
			speed[j][i] = s;
			retro[j][i] = ( s < 0 );
		}
	}

	if ( preferVedic ) // map rahu and ketu
	{
		for( i = 0; i <= nb_days; i++ )
		{
			nodeindex = ( config->iLunarNodeMode == LUNAR_NODE_MEAN ? OMEANNODE : OTRUENODE );
			nodelen = len[nodeindex][i] ;
			nodedir = retro[nodeindex][i];
			len[OMEANNODE][i] = nodelen;
			speed[OMEANNODE][i] = speed[nodeindex][i];
			len[OTRUENODE][i] = red_deg( nodelen + 180 );
			speed[OTRUENODE][i] = speed[nodeindex][i];
			retro[OMEANNODE][i] = retro[OTRUENODE][i] = nodedir;
		}
	}

	for( i = 0; i <= nb_days; i++ ) // calc rasi and nakshatra values
	{
		for( j = 0; j <= OPOSEIDON; j++ )
		{
			if ( preferVedic && ( j >= OCUPIDO || ( j > OSATURN && j < OMEANNODE ))) continue;
			rasi[j][i]      = getRasi(      len[j][i] );
			nakshatra[j][i] = getNakshatra( len[j][i], N27 );
		}
	}
	clen = true;
}

/*****************************************************
**
**   EphemExpert   ---   calcKP 
**
******************************************************/
void EphemExpert::calcKP( const int &dasa )
{
	KPEvent *e;
	vector<KPEvent>::iterator iter;

	Calculator *calculator = Session::get()->getCalculator();
	DasaExpert *expert = Session::get()->getDasaExpert( dasa );
	if ( ! expert->hasKpFeatures() || dasa == 2 ) return;

	if ( !clen  ) calcLength();
	kpevents = expert->getKPEventList( len[1][0], len[1][nb_days], jd[0] ); // calculates all lord/sublord events

	for( iter = kpevents.begin(); iter != kpevents.end(); iter++ )
	{
		e = (KPEvent*)(&(*iter));
		d->setDate( e->jd );
		e->jd = calculator->calcPlanetaryEvent( d, e->len, 1, preferVedic ); // get the dates for the events
	}
	ckp = true;
	mydasa = dasa;
}

/*****************************************************
**
**   EphemExpert   ---   writeKp
**
******************************************************/
void EphemExpert::writeKp( Writer *writer, const int &dasa )
{
	KPEvent *e;
	int d, m, y, lord, wd;
	vector<KPEvent>::iterator iter;
	wxString s;
	vector<wxString> v;
	Table *table = 0;
	DasaExpert *expert = Session::get()->getDasaExpert( dasa );
	int dasatype = expert->getType();
	if ( ! expert->hasKpFeatures() || dasa == 2 )
	{
		s.Printf( _( "K.P. events not supported for %s Dasa." ), expert->getName());
		writer->writeLine( s );
		return;
	}

	Lang *lang = Lang::get();
	Formatter *formatter = Formatter::get();
	if ( ! preferVedic )
	{
		writer->writeLine( _( "Not supported in western mode." ));
		return;
	}
	if ( ! ckp || mydasa != dasa ) calcKP( dasa );
	writeHeaderInfo( writer );

	s.Printf( _( "%s Dasa" ), expert->getName());
	writer->writeParagraph( s );

	lord = -1;
	int line = 1;
	int index = -1;
	for( iter = kpevents.begin(); iter != kpevents.end(); iter++ )
	{
		e = (KPEvent*)(&(*iter));
		if ( e->dasaindex != index )
		{
			index = e->dasaindex;
			if ( table )
			{
				if ( dasatype == DASA_TYPE_PLANET ) writer->writeHeader2( lang->getObjectName( lord, TLARGE ));
				else writer->writeHeader2( lang->getSignName( lord, TLARGE ));
				writer->writeTable( *table );
				delete table;
				table = 0;
			}
			table = new Table ( 5, 10 );
			table->setHeader( 0, _( "Day" ));
			table->setHeader( 1, _( "Time" ));
			table->setHeader( 2, _( "Lord" ));
			table->setHeader( 3, _( "Sublord" ));
			table->setHeader( 4, _( "Lunar Position" ));
			line = 1;
		}
		assert( table );
		formatter->getDateIntsFromJD( e->jd  + tz/24, d, m, y );
    wd = ((int)(e->jd + tz/24 + 1.5 )) % 7;
		assert( wd >= 0 && wd < 7 );
		s.Printf( wxT( "%02d %s" ), d, wxString( Lang::get()->getWeekdayName( wd )).Left(2).c_str() );
		//s.Printf( wxT( "%02d.%02d.%04d %s" ), d, m, y, wxString( Lang::get()->getWeekdayName( wd )).Left(2).c_str() );
		table->setEntry( 0, line, s );

		table->setEntry( 1, line, formatter->getTimeFormated( getTimeFromJD( e->jd  + tz/24 )) );
		if ( dasatype == DASA_TYPE_PLANET ) table->setEntry( 2, line, lang->getObjectName( e->lord, TLARGE ));
		else table->setEntry( 2, line, lang->getSignName( e->lord, TLARGE ));
		if ( dasatype == DASA_TYPE_PLANET ) table->setEntry( 3, line, lang->getObjectName( e->sublord, TLARGE ));
		else table->setEntry( 3, line, lang->getSignName( e->sublord, TLARGE ));

		table->setEntry( 4, line, formatter->getPosFormatted( e->len, false ) );

		lord = e->lord;
		line++;
	}
	if ( table )
	{
		if ( dasatype == DASA_TYPE_PLANET ) writer->writeHeader2( lang->getObjectName( e->lord, TLARGE ));
		else writer->writeHeader2( lang->getSignName( e->lord, TLARGE ));
		writer->writeTable( *table );
		delete table;
		table = 0;
	}
}

/*****************************************************
**
**   EphemExpert   ---   calcIngress 
**
******************************************************/
void EphemExpert::calcIngress( const int filter )
{
	int i, j, endp;

	if ( !clen  ) calcLength();
	ingressEvents.clear();

	endp = preferVedic ? OTRUENODE : OPOSEIDON;
	for( i = 0; i < nb_days; i++ )
	{
		d->setDate( i+1, month, year, -tz + 12 );
		for( j = 0; j <= endp; j++ )
		{
			if ( preferVedic ) // map nodes in vedic mode
			{
			}
			else // skip one node in western mode
			{
				if ( j == OMEANNODE && config->wLunarNodeMode == LUNAR_NODE_TRUE ) continue;
				if ( j == OTRUENODE && config->wLunarNodeMode == LUNAR_NODE_MEAN ) continue;
			}
			testIngressEvent( rasi[j][i], rasi[j][i+1], j, 0, retro[j][i] );
			if ( preferVedic )
			{
				testIngressEvent( nakshatra[j][i], nakshatra[j][i+1], j, 1, retro[j][i] );
			}
		}
	}
	ingressEvents.sort( IngressEventSort() );
	cingress = true;
}

/*****************************************************
**
**   EphemExpert   ---   testIngressEvent 
**
******************************************************/
void EphemExpert::testIngressEvent( const int& t1, const int &t2, const int &planet, const int &type, const bool &r )
{
	double tjd, targetlen;
	if ( t1 != t2 )
	{
		Calculator *calculator = Session::get()->getCalculator();
		targetlen = ( r ? t1 : t2 ) * ( type ? 13.3333333333 : 30 );
		tjd = calculator->calcPlanetaryEvent( d, targetlen, planet, preferVedic );
		ingressEvents.push_back( IngressEvent( tjd, planet, t2, type ));
	}
}

/*****************************************************
**
**   EphemExpert   ---   writeHeaderInfo
**
******************************************************/
void EphemExpert::writeHeaderInfo( Writer *writer )
{
	Lang *lang = Lang::get();
	wxString s;
	s.Printf( wxT ( "%s %d" ), lang->getMonthName( month-1 ).c_str(), year );
	writer->writeHeader1( s );
	if ( tz ) s.Printf( wxT( "%s %c%.1f" ), _( "Time Zone: UT" ), ( tz > 0 ? '+' : '-' ), tz );
	else s = _( "Time Zone: UT" );
	writer->writeParagraph( s );
}

/*****************************************************
**
**   EphemExpert   ---   writeIngress 
**
******************************************************/
void EphemExpert::writeIngress( Writer *writer, const int filter )
{
	Formatter *formatter = Formatter::get();
	Lang *lang = Lang::get();
	IngressEvent *e;
	int d, m, y, wd;
	list<IngressEvent>::iterator iter;
	wxString s, buf, pname;

	if ( ! cingress ) calcIngress( filter );
	writeHeaderInfo( writer );

	Table table( 4, ingressEvents.size()+1 );
	table.setHeader( 0, _( "Day" ));
	table.setHeader( 1, _( "Time" ));
	table.setHeader( 2, _( "Planet" ));
	table.setHeader( 3, _( "Ingress" ));

	int line = 1;
	for( iter = ingressEvents.begin(); iter != ingressEvents.end(); iter++ )
	{
		e = (IngressEvent*)(&(*iter));
		if ( e->type == 0 )
			buf = lang->getSignName( e->which, TLARGE );
		else
			buf = lang->getNakshatraName( e->which, N27, TLARGE );

		if ( preferVedic && e->planet > OSATURN )
		{
			if ( e->planet == OMEANNODE ) pname = lang->getObjectName( IRAHU, TLARGE );
			else if ( e->planet == OTRUENODE ) pname = lang->getObjectName( IKETU, TLARGE );
			else assert( 0 );
		}
		else
		{
			pname = lang->getOListObjectName( e->planet, TLARGE );
		}
		formatter->getDateIntsFromJD( e->jd  + tz/24, d, m, y );
    wd = ((int)(e->jd + tz/24+ 1.5 )) % 7;
		assert( wd >= 0 && wd < 7 );
		//s.Printf( wxT( "%02d %s" ), d, day[wd] );
		s.Printf( wxT( "%02d %s" ), d, wxString( Lang::get()->getWeekdayName( wd )).Left(2).c_str() );
		table.setEntry( 0, line, s );

		table.setEntry( 1, line, formatter->getTimeFormated( getTimeFromJD( e->jd  + tz/24 )) );
		table.setEntry( 2, line, pname );
		table.setEntry( 3, line, buf );
		line++;
	}
	writer->writeTable( table );
}

/*****************************************************
**
**   EphemExpert   ---   addLunarEvent
**
******************************************************/
void EphemExpert::addLunarEvent( const int &stdate, const double& target, const double& sdiff, const double &ediff )
{
	double slen, mlen, cjd;
	Calculator *calculator = Session::get()->getCalculator();

	d->setDate( jd[stdate] + ( target - sdiff ) / ( ediff - sdiff ));

	cjd = calculator->calcSunMoonEvent( d, target, slen, mlen );
	double aya = calculator->calcAyanamsa( cjd, config->iAyanamsa );
	lunarEvents.push_back( LunarEvent( target, cjd,
		preferVedic ? red_deg( slen - aya ) : slen,
		preferVedic ? red_deg( mlen - aya ) : mlen ));
}


/*****************************************************
**
**   EphemExpert   ---   addLunarSpecialEvent
**
******************************************************/
void EphemExpert::addLunarSpecialEvent( const int &stdate, const double& target, const double& sdiff, const double &ediff )
{
	if ( sdiff < target && ediff >= target ) addLunarEvent( stdate, target, sdiff, ediff );
}

/*****************************************************
**
**   EphemExpert   ---   calcLunar
**
******************************************************/
void EphemExpert::calcLunar()
{
	int i, t, t2;
	double diff, diff2;
	lunarEvents.clear();

	if ( !clen  ) calcLength();

	diff = red_deg( len[OMOON][0] - len[OSUN][0] );
	t = (int)(diff/12);
	for( i = 1; i <= nb_days; i++ )
	{
		diff2 = red_deg( len[OMOON][i] - len[OSUN][i] );
		t2 = (int)(diff2/12);

		if ( t2 != t )
		{
			addLunarEvent( i-1, t * 12, diff, diff2 );

			// Happens once a month, i.e. double step
			if ( t2 - t > 1 )
			{
				addLunarEvent( i-1, ( t + 1 ) * 12, diff, diff2 );
			}
		}
		// Squares
		addLunarSpecialEvent( i-1, 90, diff, diff2 );
		addLunarSpecialEvent( i-1, 270, diff, diff2 );

		// Semi Squares
		addLunarSpecialEvent( i-1, 45, diff, diff2 );
		addLunarSpecialEvent( i-1, 135, diff, diff2 );
		addLunarSpecialEvent( i-1, 225, diff, diff2 );
		addLunarSpecialEvent( i-1, 315, diff, diff2 );

		diff = diff2;
		t = t2;
	}

	lunarEvents.sort( LunarEventSort() );
	clunar = true;
}

/*****************************************************
**
**   EphemExpert   ---   writeLunar
**
******************************************************/
void EphemExpert::writeLunar( Writer *writer )
{
	if ( ! clunar ) calcLunar();
	Formatter *formatter = Formatter::get();
	Lang *lang = Lang::get();
	int d, m, y, wd;
	wxString s;
	list<LunarEvent>::iterator iter;
	LunarEvent *e;

	writeHeaderInfo( writer );
	Table table( 7, lunarEvents.size()+1 );
	table.setHeader( 0, _( "Day" ));
	table.setHeader( 1, _( "Time" ));
	table.setHeader( 2, _( "Sun" ));
	table.setHeader( 3, _( "Moon" ));
	table.setHeader( 4, _( "Angle" ));
	table.setHeader( 5, _( "Tithi" ));
	table.setHeader( 6, _( "Western" ));

	int line = 1;
	for( iter = lunarEvents.begin(); iter != lunarEvents.end(); iter++ )
	{
		e = (LunarEvent*)(&(*iter));

		formatter->getDateIntsFromJD( e->jd  + tz/24, d, m, y );
    wd = ((int)(e->jd +tz/24 + 1.5 )) % 7;
		assert( wd >= 0 && wd < 7 );
		//s.Printf( wxT( "%02d %s" ), d, day[wd] );
		s.Printf( wxT( "%02d %s" ), d, wxString( Lang::get()->getWeekdayName( wd )).Left(2).c_str() );
		table.setEntry( 0, line, s );

		table.setEntry( 1, line, formatter->getTimeFormated( getTimeFromJD( e->jd  + tz/24 )) );
		table.setEntry( 2, line, formatter->getPosFormatted( e->slen, false ) );
		table.setEntry( 3, line, formatter->getPosFormatted( e->mlen, false ) );

		int angle = (int)( red_deg( e->mlen - e->slen ) + .00001 );
		if ( angle >= 360 ) angle -= 360;
		s.Printf( wxT( "%d" ), angle );
		table.setEntry( 4, line, s );

		if ( ! ( angle % 12 ))
		{
			table.setEntry( 5, line, lang->getTithiName( angle / 12 ) );
		}
		if ( angle == 0 ) table.setEntry( 6, line, _( "New Moon" ));
		else if ( angle == 180 ) table.setEntry( 6, line, _( "Full Moon" ));
		else if ( angle == 60 || angle == 300 ) table.setEntry( 6, line, _( "Sextile" ));
		else if ( angle == 90 ) table.setEntry( 6, line, _( "Half Moon (Waxing)" ));
		else if ( angle == 120 || angle == 240 ) table.setEntry( 6, line, _( "Trine" ));
		else if ( angle == 270 ) table.setEntry( 6, line, _( "Half Moon (Waning)" ));

		// Semi squares
		else if (( angle == 45 || angle == 135 || angle == 225 || angle == 315 ))
			table.setEntry( 6, line, _( "Semi Square" ));
		line++;
	}
	writer->writeTable( table );
}

/*****************************************************
**
**   EphemExpert   ---   calcDetails 
**
******************************************************/
void EphemExpert::calcDetails()
{
	int i;
	Calculator *calculator = Session::get()->getCalculator();
	double mlen, slen, dummy, diff;

	for( i = 0; i < nb_days; i++ )
	{
		d->setDate( i+1, month, year, 0 ); // Must be 0, correct offset will follow in formatting

		// Only the time will be affected: location ist default! (not 0 - 0)
		st[i] = calculator->calcSiderealTime( d->getJD(), d->getLocation()->getLongitude() );
		calculator->calcSunRiseSunSet( d, sunrise[i], sunset[i] );

		d->setDate( sunrise[i] );
		calculator->calcPosition( d, IMOON, mlen, dummy, false, false );
		calculator->calcPosition( d, ISUN, slen, dummy, false, false );
		diff = red_deg( mlen - slen );
		tithi[i] = (int)(diff/12);
	}
	
	cdetails = true;
}

/*****************************************************
**
**   EphemExpert   ---   writeDetails 
**
******************************************************/
void EphemExpert::writeDetails( Writer *writer )
{
	int i;
	double tzdelta;
	wxString s, tz_str;
	Lang *lang = Lang::get();
	Formatter *formatter = Formatter::get();

	if ( ! cdetails ) calcDetails();
	tzdelta = ( tz ? tz/24 : 0 );

	s.Printf( wxT ( "%s %d" ), lang->getMonthName( month-1 ).c_str(), year );
	writer->writeHeader1( s );
	if ( tz ) s.Printf( wxT( "%s %c%.1f" ), _( "Time Zone: UT" ), ( tz > 0 ? '+' : '-' ), tz );
	else s = _( "Time Zone: UT" );
	writer->writeLine( s );
	s.Printf( wxT ( "%s: %s" ), _( "Location" ), d->getLocation()->getLocName().c_str()  );
	writer->writeParagraph( s );

	int line = 1;
	int nb_leaps = 0;
  for( i = 0; i < nb_days; i++ )
  {
    if ( i > 0 && weekday[i] == 0 ) nb_leaps++;
	}
	Table table( 5, nb_days+1+nb_leaps );
	table.setHeader( 0, _( "Day" ));
	table.setHeader( 1, _( "Sidereal Time" ));
	table.setHeader( 2, _( "Sunrise" ));
	table.setHeader( 3, _( "Sunset" ));
	table.setHeader( 4, _( "Tithi (Sunrise)" ));

	line = 1;
  for( i = 0; i < nb_days; i++ )
  {
    // blank line on weekend
    if ( i > 0 && weekday[i] == 0 )
		{
			for( int j = 0; j < 5; j++ ) table.setHeaderEntry( j, line, wxEmptyString );
			line++;
		}

		//s.Printf( wxT( "%02d %s" ), i+1, day[weekday[i]] );
		s.Printf( wxT( "%02d %s" ), i+1, wxString( Lang::get()->getWeekdayName( weekday[i] )).Left(2).c_str() );
		table.setEntry( 0, line, s );

		table.setEntry( 1, line, formatter->getTimeFormated( st[i] ) );
		table.setEntry( 2, line, formatter->getTimeFormated( getTimeFromJD( sunrise[i] + tzdelta )));
		table.setEntry( 3, line, formatter->getTimeFormated( getTimeFromJD( sunset[i] + tzdelta )));

		table.setEntry( 4, line, lang->getTithiName( tithi[i] ));
		line++;
	}
	writer->writeTable( table );
}

/*****************************************************
**
**   EphemExpert   ---   getPlanetaryIndex 
**
******************************************************/
int EphemExpert::getPlanetaryIndex( const int &index )
{
	if ( preferVedic )
	{
		switch( index )
		{
			case IMOON: return OMOON; break;
			case IMARS: return OMARS; break;
			case IMERCURY: return OMERCURY; break;
			case IJUPITER: return OJUPITER; break;
			case IVENUS: return OVENUS; break;
			case ISATURN: return OSATURN; break;
			case IRAHU: return OMEANNODE; break;
			case IKETU: return OTRUENODE; break;
			default: return OSUN; break; // sun
		}
	}
	else
	{
		if ( index <= WPLUTO ) return index - WSUN;
		else if ( index == WLUNARNODE ) return ( config->wLunarNodeMode == LUNAR_NODE_MEAN ? OMEANNODE : OTRUENODE);
		else return ( index - 24 );
	}
}

/*****************************************************
**
**   EphemExpert   ---   getLength
**
******************************************************/
double EphemExpert::getLength( const int &planet, const int &day )
{
	return len[getPlanetaryIndex(planet)][day];
}

/*****************************************************
**
**   EphemExpert   ---   getSpeed
**
******************************************************/
double EphemExpert::getSpeed( const int &planet, const int &day )
{
	return speed[getPlanetaryIndex(planet)][day];
}

/*****************************************************
**
**   EphemExpert   ---   getRetro
**
******************************************************/
bool EphemExpert::getRetro( const int &planet, const int &day )
{
	return retro[getPlanetaryIndex(planet)][day];
}

/*****************************************************
**
**   EphemExpert   ---   calcMonth
**
******************************************************/
void EphemExpert::calcMonth()
{
	if ( !clen  ) calcLength();
}

/*****************************************************
**
**   EphemExpert   ---   writeDefaultEphemeris
**
******************************************************/
void EphemExpert::writeDefaultEphemeris( Writer *writer, const int filter )
{
	wxString tz_str, s, rasi_str, nak_str;
	int p, i, j, deg, min, startp, endp, numcols;
	double rasilen;
	bool showrasi, shownak;
	wxString d;
	Lang *lang = Lang::get();

	if ( !clen  ) calcLength();
	writeHeaderInfo( writer );

	if ( preferVedic )
	{
		startp = ISUN;
		endp = IKETU;
		numcols = endp - startp + 2;
	}
	else
	{
		startp = ( filter == 2 ? WCUPIDO : WSUN );
		endp = ( filter == 1 ?  WLUNARNODE : WPOSEIDON );
		numcols = ( filter == 0 ?  endp - startp : endp - startp + 2 );
	}

	int line = 1;
	int nb_leaps = 0;
  for( i = 0; i < nb_days; i++ )
  {
    if ( i > 0 && weekday[i] == 0 ) nb_leaps++;
	}
	Table table( numcols, nb_days+nb_leaps+2 );
	table.setHeader( 0, _( "Day" ));

	// Header
	int col = 1;
	for( j = startp; j <= endp; j++ )
	{
		if ( j > WLUNARNODE && j < WCUPIDO ) continue; // loop for AS and MC

		table.setHeader( col, lang->getObjectName( j, TLARGE ));
		col++;
	}
	line = 1;
  for( i = 0; i < nb_days; i++ )
  {
    //if ( i > 0 && weekday[i] == 0 ) o << Endl; // blank line on weekend
    if ( i > 0 && weekday[i] == 0 )
		{
			for( j = 0; j < numcols; j++ ) table.setHeaderEntry( j, line, wxEmptyString );
			line++;
		}

		//s.Printf( wxT( "%02d %s" ), i+1, day[weekday[i]] );
		s.Printf( wxT( "%02d %s" ), i+1, wxString( Lang::get()->getWeekdayName( weekday[i] )).Left(2).c_str() );
		table.setEntry( 0, line, s );

		col = 1;
		for( j = startp; j <= endp; j++ )
		{
			if ( j > WLUNARNODE && j < WCUPIDO ) continue; // loop for AS and MC
			p = getPlanetaryIndex( j );

			rasilen = getRasiLen( len[p][i] ) + .008333333;
			deg = (int)rasilen;
			min = (int)( 60 * ( rasilen - (double)deg ));

			// symbol for retrogression
			d = wxT( " " );
			if ( retro[p][i] ) { d = _( "R" ); }
			else if ( i > 1 && retro[p][i] != retro[p][i-1] ) { retro[p][i] ? d = _( "R" ): d = _( "D" ); }
			else if ( i == nb_days - 1 && retro[p][i] ) { d = _( "R" ); }

			showrasi = shownak = true;
			rasi_str = showrasi ? lang->getSignName( rasi[p][i] ) : wxT( "  " );
			nak_str = shownak ? lang->getNakshatraName( nakshatra[p][i], N27, TSHORT ) : wxT( "   " );

			// ready to print now
			if ( config->useVedicPositions )
			{
				if ( preferVedic )
				{
					s.Printf( wxT( "%s%02d-%02d-%02d %s" ), d.c_str(), rasi[p][i], deg, min, nak_str.c_str() );
				}
				else
				{
					s.Printf( wxT( "%s%02d-%02d-%02d" ), d.c_str(), rasi[p][i], deg, min );
				}
			}
			else
			{
				if ( preferVedic )
				{
					s.Printf( wxT( "%s%02d%s%02d %s" ), d.c_str(), deg, rasi_str.c_str(), min, nak_str.c_str() );
				}
				else
				{
					s.Printf( wxT( "%s%02d%s%02d" ), d.c_str(), deg, rasi_str.c_str(), min );
				}
			}
			table.setEntry( col, line, s );
			
			col++;
		}

		line++;
	}
	// Header
	table.setHeaderEntry( 0, line, _( "Day" ));
	col = 1;
	for( j = startp; j <= endp; j++ )
	{
		if ( j > WLUNARNODE && j < WCUPIDO ) continue; // loop for AS and MC

		table.setHeaderEntry( col, line, lang->getObjectName( j, TLARGE ));
		col++;
	}
	writer->writeTable( table );
}


