/*

	This is the source code of

   	M A I T R E Y A
    ===============

	Open source platform for Vedic and western astrology

  File           Hora.cpp
  Release        4.1
  Author         Martin Pettau
  Copyright (C)  2003-2006 by the author

  Released under the Artistic License as published by the
  Free Software Foundation, read the file 'COPYING' for more information.

*/

#ifdef __GNUG__
	#pragma implementation "Hora.h"
#endif

#include "Hora.h"
#include "Calculator.h"
#include "constants.h"
#include "DataSet.h"
#include "func.h"
#include "Lang.h"
#include "Session.h"
#include "Writer.h"

#include <wx/string.h>

/*****************************************************
**
**   HoraExpert   ---   Constructor 
**
******************************************************/
HoraExpert::HoraExpert( DataSet *ds )
{
	dataset = new DataSet( *ds );
}

/*****************************************************
**
**   HoraExpert   ---   Destructor 
**
******************************************************/
HoraExpert::~HoraExpert()
{
	delete dataset;
}

//#define HORA_DEB
/*****************************************************
**
**   HoraExpert   ---   update 
**
******************************************************/
void HoraExpert::update( const double &jd )
{
	int i, wd;
	const int lordseq[7] = { ISATURN, IJUPITER, IMARS, ISUN, IVENUS, IMERCURY, IMOON };
	const int rev_lordseq[7] = { 3, 6, 2, 5, 1, 4, 0 };
	double sunrise, sunset, sunrise_next, dummy, thejd;
	Calculator *calculator = Session::get()->getCalculator();

	thejd = jd;
	dataset->setDate( thejd );
	calculator->calcSunRiseSunSet( dataset, sunrise, sunset );
	if ( sunrise > thejd )
	{
		thejd--;
		dataset->setDate( thejd );
		calculator->calcSunRiseSunSet( dataset, sunrise, sunset );
	}

	dataset->setDate( thejd + 1 );
	calculator->calcSunRiseSunSet( dataset, sunrise_next, dummy );
	dataset->setDate( jd );

	wd = (int)(sunrise + 1.5 ) % 7;

#ifdef HORA_DEB
	wxChar tmp[256];
	cout << "##################  DEBUG  ######################" << Endl;
	cout << "Dataset:" << Endl;
	dataset->dump( cout );
	cout << "WEEKDAY " << wd << Endl;
	Formatter *formatter = Formatter::get();
	formatter->getFullDateStringFromJD( tmp, sunrise );
	cout << "sunrise: " << tmp << Endl;
	formatter->getFullDateStringFromJD( tmp, jd );
	cout << "jd: " << tmp << Endl;
	formatter->getFullDateStringFromJD( tmp, thejd );
	cout << "tmpjd: " << tmp << Endl;
	formatter->getFullDateStringFromJD( tmp, sunset );
	cout << "sunset: " << tmp << Endl;
	formatter->getFullDateStringFromJD( tmp, sunrise_next );
	cout << "sunrise NEXT: " << tmp << Endl;
	cout << "##################  DEBUG  ######################" << Endl;
#endif

	for( i = 0; i < 24; i++ )
	{
		horaLord[i] = lordseq[ ( rev_lordseq[wd] +i ) % 7 ];
	}
	
	double daydur = sunset - sunrise;
	double daystep = daydur / 12;

	double nightdur = sunrise_next - sunset;
	double nightstep = nightdur / 12;

	horaStart[24] = sunrise_next;
	for ( i = 0; i < 12; i++ )
	{
		horaStart[i] = sunrise + i * daystep;
		horaStart[i+12] = sunset + i * nightstep;
	}
}

/*****************************************************
**
**   HoraExpert   ---   getCurrentHoraLord 
**
******************************************************/
void HoraExpert::getCurrentHoraLord( const double &jd, int *lord, int *sublord, double *percent )
{
	int i;
	for( i = 0; i < 24; i++ )
	{
		if ( jd > horaStart[i] && jd < horaStart[i+1] )
		{
			*lord = horaLord[0];
			*sublord = horaLord[i];
			*percent = 100.0 * ( jd - horaStart[i] ) / ( horaStart[i+1] - horaStart[i] );
			return;
		}
	}
	// else
	*lord = *sublord = -1;
	*percent = 0;
}

/*****************************************************
**
**   HoraExpert   ---   getHoraLord 
**
******************************************************/
int HoraExpert::getHoraLord( const int &i )
{
	assert( i >= 0 && i < 24 );
	return horaLord[i];
}

/*****************************************************
**
**   HoraExpert   ---   getHoraStart 
**
******************************************************/
double HoraExpert::getHoraStart( const int &i )
{
	assert( i >= 0 && i < 25 );
	return horaStart[i];
}

/*****************************************************
**
**   HoraExpert   ---   write
**
******************************************************/
void HoraExpert::write( Writer *writer, const bool printactual )
{
	int i, lord, sublord;
	double percent;
	wxString s;

	Formatter *formatter = Formatter::get();
	double corr = ( dataset->getLocation()->getTimeZone() + dataset->getLocation()->getDST()) / 24;
  writer->writeHeader1( _( "Hora" ));
	s.Printf( wxT( "%s: %s" ), _( "Lord of the day" ), Lang::get()->getObjectName( horaLord[0], TLARGE ).c_str());
  writer->writeLine( s );
	s.Printf( wxT( "%s: %s" ), _( "Sunrise" ), formatter->getTimeFormated( getTimeFromJD( horaStart[0] + corr )).c_str() );
  writer->writeLine( s );
	s.Printf( wxT( "%s: %s" ), _( "Sunset" ), formatter->getTimeFormated( getTimeFromJD( horaStart[12] + corr )).c_str() );
  writer->writeParagraph( s );

	Table table( 7, 13 );
	table.setHeader( 0,  _( "Begin" ));
	table.setHeader( 1,  _( "Lord" ));
	table.setHeader( 2,  _( "End" ));
	table.setHeader( 3,  wxEmptyString );
	table.setHeader( 4,  _( "Begin" ));
	table.setHeader( 5,  _( "Lord" ));
	table.setHeader( 6,  _( "End" ));
	int line = 1;
	int colskip = 0;
	for ( i = 0; i < 24; i++ )
	{
		if ( i == 12 )
		{
			colskip = 4;
			line = 1;
		}
		table.setEntry( 3, line, wxEmptyString );
		table.setEntry( colskip, line, formatter->getTimeFormated( getTimeFromJD( horaStart[i] + corr )) );
		table.setEntry( 1+colskip, line, Lang::get()->getObjectName( horaLord[i], TLARGE ));
		table.setEntry( 2+colskip, line, formatter->getTimeFormated( getTimeFromJD( horaStart[i+1] + corr )) );
		line++;
	}
	writer->writeTable( table ); 
	if ( printactual )
	{
		getCurrentHoraLord( dataset->getJD(), &lord, &sublord, &percent )	;
		s.Printf( wxT( "%s: %s / %s %02.2f%%" ), _( "Current Lord" ), Lang::get()->getObjectName( lord, TMEDIUM ).c_str(),
				Lang::get()->getObjectName( sublord, TMEDIUM ).c_str(), percent );
		writer->writeParagraph( s ); 
	}
}

