/*

	This is the source code of

   	M A I T R E Y A
    ===============

	Open source platform for Vedic and western astrology

  File           Horoscope.cpp
  Release        4.1
  Author         Martin Pettau
  Copyright (C)  2003-2006 by the author

  Released under the Artistic License as published by the
  Free Software Foundation, read the file 'COPYING' for more information.

*/

#ifdef __GNUG__
	#pragma implementation "Horoscope.h"
#endif

#include "Horoscope.h"

#include <wx/string.h>
#include <math.h>

#include "func.h"
#include "Calculator.h"
#include "Conf.h"
#include "IPlanet.h"
#include "Jaimini.h"
#include "Lang.h"
#include "Nakshatra.h"
#include "Session.h"
#include "Varga.h"
#include "VimsottariDasa.h"
#include "Writer.h"

extern const wxChar *chara_karaka_name[8];
extern const wxChar *chara_karaka_name_long[8];
extern Config *config;

/*****************************************************
**
**   Horoscope   ---   Constructor 
**
******************************************************/
Horoscope::Horoscope()
 : SimpleHoroscope()
{
	int i, j;

	for ( i = 0; i < NUM_LAGNA; i++ ) lagna_len[i] = 0;
	for ( i = 0; i < NUM_UPA; i++ ) upagraha_len[i] = 0;

	for( i = 0; i < MAX_IOBJECTS_LAGNA_UPA; i++ )
		for( j = 0; j < NB_VARGAS; j++ )
			varga[i][j] = new Varga( i, j );
	
	for( i = 0; i < MAX_IOBJECTS_LAGNA_UPA; i++ )
		iplanets.push_back( new IPlanet( this, i ));
}

/*****************************************************
**
**   Horoscope   ---   Destructor 
**
******************************************************/
Horoscope::~Horoscope()
{
	int i, j;
	unsigned int i1;
	for( i1 = 0; i1 < iplanets.size(); i1++ ) delete iplanets[i1];
	for( i = 0; i < MAX_IOBJECTS_LAGNA_UPA; i++ )
		for( j = 0; j < NB_VARGAS; j++ )
			delete varga[i][j];
}

/*****************************************************
**
**   Horoscope   ---   update 
**
******************************************************/
void Horoscope::update()
{
	int i, j, lord;
	Session *session = Session::get();
	Calculator *calculator = session->getCalculator();
	VargaExpert expert( this );

	updatePlanets();
	isDayBirth = calculator->calcSunRiseSunSet( dataSet, sunrise, sunset );

	// Upagraha Update
	double lsun = getObjectLength( ISUN );
	// Dhuma
	upagraha_len[ODHUMA] = red_deg( lsun + 133.333333333 );
	// Vyatipata
	upagraha_len[OVYATIPATA] = red_deg( 360 - 133.33333333 - lsun );
	// Parivesha
	upagraha_len[OPARIVESHA] = red_deg( 180 - 133.33333333 - lsun );
	// Chapa
	upagraha_len[OCHAPA] = red_deg( lsun + 313.333333333 );
	// Upaketu
	upagraha_len[OUPAKETU] = red_deg( lsun + 330 );

	// Upagrahas from kala
	double day_len = fabs( sunset - sunrise );
	day_len /= 8;

	double thejd, asc;
	for( i = 0; i < 8; i++ )
	{
		if ( isDayBirth )
		{
			lord = ( dataSet->getWeekDay() + i ) % 8;
			thejd = sunrise + i * day_len;
		}
		else
		{
			lord = ( dataSet->getWeekDay() + i + 4 ) % 8; 
			thejd = sunset + i * day_len;
		}

		asc = red_deg( session->getCalculator()->calcAscendant( thejd, getLocation()->getLatitude(),
					getLocation()->getLongitude() ) - iayanamsa );
		
		switch( lord )
		{
			case ISUN:
				upagraha_len[OKALA] = asc;
			break;
			case IMARS:
				upagraha_len[OMRITYU] = asc;
			break;
			case IMERCURY:
				upagraha_len[OARDHAPRAHARA] = asc;
			break;
			case IJUPITER:
				upagraha_len[OYAMAGHANTAKA] = asc;
			break;
			case ISATURN:
				upagraha_len[OGULIKA] = asc;
			break;
			default:
				//cout << "DEFAULT " << lord << Endl;
			break;
		}
	}

	// Special Lagnas
	lagna_len[OBHAVA_LAGNA] = red_deg( 360 * ( getJD() - sunrise ) + lsun);
	lagna_len[OHORA_LAGNA] = red_deg( 720 * ( getJD() - sunrise ) + lsun);
	lagna_len[OGHATIKA_LAGNA] = red_deg( 1800 * ( getJD() - sunrise ) + lsun);

	double l;
	for( i = 0; i < MAX_IOBJECTS_LAGNA_UPA; i++ )
	{
		l = getObjectLength(i);
		iplanets[i]->length = l;
		iplanets[i]->isRetro = isObjectRetrograde(i);
		for( j = 0; j < expert.getNbVargas(); j++ )
			{ expert.calcVarga( varga[i][j], l, j ); }
	}
	for( i = 0; i <= IKETU; i++ )
	{
		for( j = 0; j < expert.getNbVargas(); j++ )
			{ expert.calcBala( i, getVarga( i, j )->getLength(), j ); }
		iplanets[i]->update();
	}

	// Avasthas
	updateAvasthas();
}

/**************************************************************
***
** Horoscope   ---   getVarga
***
***************************************************************/
Varga *Horoscope::getVarga( const int &planet, const int &v )
{
	assert ( planet >= 0 && planet < MAX_IOBJECTS_LAGNA_UPA  );
	assert (( v >= 0 ) && ( v < NB_VARGAS ));

	return varga[planet][v];
}

/**************************************************************
***
** Horoscope   ---   getIPlanet
***
***************************************************************/
IPlanet *Horoscope::getIPlanet( const int &i )
{
	assert( i >= 0 && i < (int)iplanets.size());

	return iplanets[i];
}

/*****************************************************
**
**   Horoscope   ---   getAvastha 
**
******************************************************/
int Horoscope::getAvastha( const int &planet, const int whichAvastha )
{
	return sayanadiAvastha[planet];
}

/*****************************************************
**
**   Horoscope   ---   getObjectLength 
**
******************************************************/
double Horoscope::getObjectLength( const int &i )
{
	double ret = 0;

	if ( i == IBHAVA_LAGNA ) ret = lagna_len[OBHAVA_LAGNA];
	else if ( i == IHORA_LAGNA ) ret = lagna_len[OHORA_LAGNA];
	else if ( i == IGHATIKA_LAGNA ) ret = lagna_len[OGHATIKA_LAGNA];

	else if (( i >= IDHUMA ) && ( i <= IGULIKA )) ret = upagraha_len[ODHUMA+i-IDHUMA];
	else
	{
		return SimpleHoroscope::getObjectLength( i );
	}

	return red_deg( ret );
}

/*****************************************************
**
**   Horoscope   ---   updateAvasthas 
**
******************************************************/
void Horoscope::updateAvasthas()
{
	//o << "Update Avasthas" << Endl;
	int janma_naks = getNakshatra(getVarga( IMOON, 0 )->getLength(), N27 ) + 1;
	for( int i = ISUN; i <= IKETU; i++ )
	{
		int ghati = (int)(( getJD() - sunrise ) * 60 + 1 );
		int naks =  getNakshatra( getVarga( i, 0 )->getLength(), N27 ) + 1;
		int navamsa = getVarga( i, 1 )->getRasi() + 1;
		int asc = getIPlanet( IASCENDANT )->getRasi() + 1;

		//o << "NAKS " << naks << " NAV " << navamsa << " janma " << janma_naks
		//	<< " ghati " << ghati << " asc " << asc << Endl;

		sayanadiAvastha[i] = ( naks * ( i + 1 ) * navamsa ) + janma_naks + ghati + asc;
		sayanadiAvastha[i] %= 12;
	}
}

/*****************************************************
**
**   Horoscope   ---   dumpAvasthas 
**
******************************************************/
void Horoscope::dumpAvasthas( wxString &o )
{
	o << wxT(  "~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~" ) << Endl;
	o << wxT(  "Avasthas" ) << Endl;
	for( int i = 0; i < 9; i++ ) o <<  Lang::get()->getObjectName(i, TMEDIUM) << wxT( ": " )
		<< getAvasthaName( sayanadiAvastha[i] ) << Endl;
	o << wxT(  "~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~" ) << Endl;
}

/*****************************************************
**
**   Horoscope   ---   writeVedicPlanets 
**
******************************************************/
void Horoscope::writeVedicPlanets( Writer *writer )
{
  VargaExpert expert;
  int i;
  Lang *lang = Lang::get();
  Calculator *calculator = Session::get()->getCalculator();
  wxString s1;
	Formatter *formatter = Formatter::get();
  //NakshatraExpert nexpert;

	VargaView view( this, 0 );
	VargaView nview( this, 1 );
	JaiminiExpert jexpert( &nview );
	jexpert.calcCharaKarakas();
  NakshatraExpert nexpert;

  writer->writeHeader1( _( "Vedic Planets" ));
	Table table( 5, 11 );
	table.setHeader( 0,  _( "Planet" ));
	table.setHeader( 1,  _( "Length" ));
	table.setHeader( 2,  _( "Karaka" ));
	table.setHeader( 3,  _( "Navamsa" ));
	table.setHeader( 4,  _( "Nakshatra" ));

	int line = 1;
  for( i = ISUN; i <= IASCENDANT; i++ )
  {
		table.setEntry( 0, line,  lang->getObjectName( i, TLARGE ));
		table.setEntry( 1, line,  formatter->getPosFormatted( getObjectLength(i), isObjectRetrograde(i)) );

    if ( i <= IRAHU )  table.setEntry( 2, line, lang->getKarakaName( jexpert.getCharaKarakaProperty(i)) );

    table.setEntry( 3, line, lang->getSignName(nview.getRasi( i ), config->signPrecision ));
    table.setEntry( 4, line, lang->getNakshatraName(view.getNakshatra(i, N27), N27, TLARGE ) );
		line++;
  }
	writer->writeTable( table ); 

  writer->writeHeader2( _( "Upagrahas" ));
	Table t2( 2, 11 );
	t2.setHeader( 0,  _( "Object" ));
	t2.setHeader( 1,  _( "Length" ));
	line = 1;
  for( i = IDHUMA; i <= IGULIKA; i++ )
  {
    t2.setEntry( 0, line, lang->getObjectName(i, TLARGE));
    t2.setEntry( 1, line, formatter->getPosFormatted( view.getLength(i), -1, DEG_PRECISION_MINUTE ) );
		line++;
  }
	writer->writeTable( t2 ); 

  writer->writeHeader2( _( "Lagnas" ));
	Table t3( 2, 4 );
	t3.setHeader( 0,  _( "Object" ));
	t3.setHeader( 1,  _( "Length" ));
	line = 1;
  for( i = IBHAVA_LAGNA; i <= IGHATIKA_LAGNA; i++ )
  {
    t3.setEntry( 0, line, lang->getObjectName(i, TLARGE));
    t3.setEntry( 1, line, formatter->getPosFormatted( view.getLength(i) ) );
		line++;
  }
	writer->writeTable( t3 ); 

  writer->writeHeader2( _( "Qualities" ));
	Table t4( 2, 5 );
	t4.setHeader( 0,  _( "Quality" ));
	t4.setHeader( 1,  _( "Value" ));
  double mlen = getObjectLength( IMOON );

	line = 1;
	t4.setEntry( 0, line, _( "Varna" ) );
	t4.setEntry( 1, line++, lang->getVarnaName( getVarna( mlen )));

	t4.setEntry( 0, line, _( "Yoni" ) );
	t4.setEntry( 1, line++, lang->getYoniName( nexpert.getYoni( mlen )));

	t4.setEntry( 0, line, _( "Gana" ) );
	t4.setEntry( 1, line++, lang->getGanaName( nexpert.getGana( mlen )));

	t4.setEntry( 0, line, _( "Nadi" ) );
	t4.setEntry( 1, line++, lang->getNadiName( nexpert.getNadi( mlen )));
	writer->writeTable( t4 ); 

  writer->writeHeader2( _( "Moon's Nakshatra and Pada Portions" ));
	Table t5( 2, 3 );
	t5.setHeader( 0,  _( "Quality" ));
	t5.setHeader( 1,  _( "Value" ));
	t5.setEntry( 0, 1, _( "Nakshatra" ) );
  double nportion = calculator->calcNakshatraPortion( getDataSet(), getObjectLength( IMOON ), false );
	s1.Printf( wxT( "%1.6f" ), nportion );
	t5.setEntry( 1, 1, s1 );
	t5.setEntry( 0, 2, _( "Pada" ) );
  double pportion = calculator->calcNakshatraPortion( getDataSet(), getObjectLength( IMOON ), true );
	s1.Printf( wxT( "%1.6f" ), pportion );
	t5.setEntry( 1, 2, s1 );
	writer->writeTable( t5 ); 
}

/*****************************************************
**
**   Horoscope   ---   writeWesternPlanets 
**
******************************************************/
void Horoscope::writeWesternPlanets( Writer *writer )
{
	wxString s;
	Formatter *f = Formatter::get();
	Lang *l = Lang::get();

  writer->writeHeader1( _( "Western Planets" ));
	Table table( 3, 34 );
	table.setHeader( 0,  _( "Planet" ));
	table.setHeader( 1,  _( "Length" ));
	table.setHeader( 2,  _( "House" ));

	int line = 1;
	for( int i = WSUN; i <= WPOSEIDON; i++ )
	{
		table.setEntry( 0, line, l->getObjectName(i, TLARGE) );
		table.setEntry( 1, line, f->getPosFormatted( getObjectLength( i ), isObjectRetrograde(i), DEG_PRECISION_SECOND ) );
		s.Printf( wxT( "%02d" ), getHousePos( i ) + 1 );
		table.setEntry( 2, line, s );
		line++;
	}
	for( int i = 0; i < 12; i++ )
	{
		s.Printf( wxT( "%02d"), i+1 );
		table.setEntry( 0, line, s );
		table.setEntry( 1, line, f->getPosFormatted( getHouse( i, false ), false ) );
		line++;
	}
	writer->writeTable( table ); 
}

/*****************************************************
**
**   Horoscope   ---   updateKP 
**
******************************************************/
void Horoscope::updateKP( const int &dasaindex )
{
	DasaExpert *expert = Session::get()->getDasaExpert( dasaindex );
	int i;
	for( i = ISUN; i <= IGULIKA; i++ )
	{
		kp_lord[i] = kp_sublord[i] = 0;
		expert->getKPLord( getObjectLength(i), kp_lord[i], kp_sublord[i] );
	}
	for( i = HOUSE1; i <= HOUSE12; i++ )
	{
		expert->getKPLord( getHouse( i, true, false ), kp_lord[i+IGULIKA], kp_sublord[i+IGULIKA] );
	}
}

/*****************************************************
**
**   Horoscope   ---   getKPLord 
**
******************************************************/
int Horoscope::getKPLord( const int &planet, const bool &sublord )
{
	assert( planet >= ISUN && planet <= MAX_IOBJECTS_LAGNA_UPA_BHAVA );
	return( sublord ?  kp_sublord[planet] : kp_lord[planet] );
}

/*****************************************************
**
**   Horoscope   ---   writeKp
**
******************************************************/
void Horoscope::writeKp( Writer *writer, const int &dasaindex )
{
	wxString s;
	Formatter *f = Formatter::get();
	Lang *lang = Lang::get();
	int i;
	Lang *l = Lang::get();
	DasaExpert *expert = Session::get()->getDasaExpert( dasaindex );

	if ( ! expert->hasKpFeatures() ) // Not supported by all (e.g. Jaimini)
	{
		writer->writeLine( _( "Not supported" ) );
		return;
	}
	updateKP( dasaindex );

  writer->writeHeader1( _( "Krishnamurti Paddhati" ));
	s.Printf( wxT( "%s: %s" ), _( "Dasa"), expert->getName());
  writer->writeParagraph( s );
	Table table( 6, 22 );
	table.setHeader( 0,  _( "Planet" ));
	table.setHeader( 1,  _( "Length" ));
	table.setHeader( 2,  _( "Sign Lord" ));
	table.setHeader( 3,  _( "Bhava" ));
	table.setHeader( 4,  _( "KP Lord" ));
	table.setHeader( 5,  _( "KP Sublord" ));

	int line = 1;
	for( i = ISUN; i <= IKETU; i++ )
	{
		table.setEntry( 0, line, l->getObjectName(i, TLARGE) );
		table.setEntry( 1, line, f->getPosFormatted( getObjectLength( i ), isObjectRetrograde(i) ) );
		table.setEntry( 2, line, Lang::get()->getObjectName( getLord((int)(getObjectLength( i )/30)), TLARGE ) );
		s.Printf( wxT( "%02d" ), getHousePos(i)+1 );
		table.setEntry( 3, line, s );
		if ( expert->getType() == DASA_TYPE_SIGN )
		{
			table.setEntry( 4, line, Lang::get()->getSignName( kp_lord[i], TLARGE ) );
			table.setEntry( 5, line, Lang::get()->getSignName( kp_sublord[i], TLARGE ) );
		}
		else
		{
			table.setEntry( 4, line, Lang::get()->getObjectName( kp_lord[i], TLARGE ) );
			table.setEntry( 5, line, Lang::get()->getObjectName( kp_sublord[i], TLARGE ) );
		}
		line++;
	}
	for( i = HOUSE1; i <= HOUSE12; i++ )
	{
		//s.Printf( wxT( "%d. %s" ), i+1, _( "Bhava" ) );
		table.setEntry( 0, line, lang->getBhavaName( i ));
		table.setEntry( 1, line, f->getPosFormatted( getHouse( i, true, false ), false ) );
		table.setEntry( 2, line, Lang::get()->getObjectName( getLord((int)(getObjectLength( i )/30)), TLARGE ) );
		if ( expert->getType() == DASA_TYPE_SIGN )
		{
			table.setEntry( 4, line, Lang::get()->getSignName( kp_lord[i+IGULIKA], TLARGE ) );
			table.setEntry( 5, line, Lang::get()->getSignName( kp_sublord[i+IGULIKA], TLARGE ) );
		}
		else
		{
			table.setEntry( 4, line, Lang::get()->getObjectName( kp_lord[i+IGULIKA], TLARGE ) );
			table.setEntry( 5, line, Lang::get()->getObjectName( kp_sublord[i+IGULIKA], TLARGE ) );
		}
		line++;
	}
	writer->writeTable( table ); 
}

/*****************************************************
**
**   Horoscope   ---   writeAspects 
**
******************************************************/
void Horoscope::writeAspects( Writer *writer )
{
	int i, j;
	wxString s;
	Lang *lang = Lang::get();
	double aspval;

  writer->writeHeader1( _( "Sphuta Drishti" ));
	Table table( 10, 8 );
	table.setHeader( 0,  wxT( "" ));
	for ( i = ISUN; i <= IKETU; i++ ) table.setHeader( i+1, lang->getObjectName( i, TMEDIUM ));

	int line = 1;
	for ( i = ISUN; i <= ISATURN; i++ )
	{
		table.setEntry( 0, line, lang->getObjectName( i, TMEDIUM ) );
		for ( j = ISUN; j <= IKETU; j++ )
		{
			aspval = getAspectValue( i, getObjectLength( i ), j, getObjectLength(j));
			s.Printf( wxT( "%02.2f " ), aspval );
			table.setEntry( j+1, line, s );
		}
		line++;
	}
	writer->writeTable( table ); 
}

/*****************************************************
**
**   TajakaHoroscope   ---   Constructor 
**
******************************************************/
TajakaHoroscope::TajakaHoroscope( DataSet *dataset )
{
	//*dataSet = *dataset;
	birth_ds = dataset;
	setHName( wxT( "Tajaka" ));
	setLocation( *dataset->getLocation() );
}


/*****************************************************
**
**   TajakaHoroscope   ---   calcTajaka 
**
******************************************************/
void TajakaHoroscope::calcTajaka( const int &syear, const bool &vedic )
{
	int dummy, month, year;
	Formatter::get()->getDateIntsFromJD( birth_ds->getJD(), dummy, month, year );
	//setDate( birthjd + ( syear - year ) * 365.25 );
	Calculator *calculator = Session::get()->getCalculator();
	double jd = calculator->calcTajakaJd( birth_ds, syear, vedic );
	setDate( jd );
}

/*****************************************************
**
**   TajakaHoroscope   ---   dump 
**
******************************************************/
void TajakaHoroscope::dump( wxString &o, const bool &vedic )
{
	wxChar tmp[256];
	wxString s;
	int i;
	Formatter *f = Formatter::get();
	Lang *l = Lang::get();
	Location *location = getLocation();

	o << wxT( "Date " ) << f->getFullDateStringFromJD( getJD() + ( location->getTimeZone() + location->getDST())/24 ) << Endl;

	if ( ! vedic )
	{
		o << _( "Western planets" ) << Endl;
		for( i = WSUN; i <= WPOSEIDON; i++ )
		{
			o << l->getObjectName(i, TMEDIUM)
				<< wxT( " : " )
				<< f->getPosFormatted( getObjectLength( i ), isObjectRetrograde(i), DEG_PRECISION_SECOND )
				<< Endl;
		}
		for( i = 0; i < 12; i++ )
		{
			s.Printf( wxT( "  %02d : " ), i+1 );
			o << s
				<< f->getPosFormatted( getHouse( i, false ), false )
				<< Endl;
		}
	}
	else
	{
		for( i = ISUN; i <= IASCENDANT; i++ )
		{
			s.Printf( wxT( "%4s : " ), l->getObjectName( i, TMEDIUM ).c_str() );
			o << s << f->getPosFormatted( getObjectLength( i ), isObjectRetrograde(i), DEG_PRECISION_SECOND ) << Endl;
		}
	}
}

/*****************************************************
**
**   CompositHoroscope   ---   Constructor 
**
******************************************************/
CompositHoroscope::CompositHoroscope()
  : Horoscope()
{
}

/*****************************************************
**
**   CompositHoroscope   ---   update 
**
******************************************************/
void CompositHoroscope::update( Horoscope *h1, Horoscope *h2 )
{
	int i, j;
	double eps, r[13];
	Calculator *calculator = Session::get()->getCalculator();
	VargaExpert expert( this );

	if ( ! h1 || ! h2 ) return;

	iayanamsa = .5 * ( h1->getIAyanamsa() + h1->getIAyanamsa());
	wayanamsa = .5 * ( h1->getWAyanamsa() + h1->getWAyanamsa());
	siderealTime = .5 * ( h1->getSiderealTime() + h2->getSiderealTime());
	for( i = 0; i < WARIES; i++ )
	{
		object_len[i] = calcGeometricalMidpoint( h1->object_len[ i ], h2->object_len[ i ] );
	}
	double arm1 = h1->getSiderealTime() * 15;
	double arm2 = h2->getSiderealTime() * 15;
	double arm = calcGeometricalMidpoint( arm1, arm2 );
	calculator->calcEps( eps, .5 * ( h1->getJD() + h2->getJD() ));
	//eps=23.21;
	calculator->calcAscendantByArmc( arm, eps, .5 * ( h1->getLatitude() + h2->getLatitude()), r );
	for( int i = 0; i < 12; i++ ) whousecusp[i] = ihousecusp[i] = r[i];

	double l;
	for( i = 0; i < MAX_IOBJECTS_LAGNA_UPA; i++ )
	{
		l = getObjectLength(i);
		iplanets[i]->length = l;
		iplanets[i]->isRetro = isObjectRetrograde(i);
		for( j = 0; j < expert.getNbVargas(); j++ )
			{ expert.calcVarga( varga[i][j], l, j ); }
	}
}

/*****************************************************
**
**   CompositHoroscope   ---   dump
**
******************************************************/
void CompositHoroscope::dump( wxString &o, const bool vedic )
{
	Formatter *f = Formatter::get();
	Lang *l = Lang::get();
	int i;

	o << wxT( "CompositHoroscope " ) << Endl;
	int starti = ( vedic ? ISUN : WSUN );
	int endi = ( vedic ? IKETU: WPOSEIDON );
	for( i = starti; i <= endi; i++ )
	{
		o <<  l->getObjectName(i, TLARGE)
			<< wxT( " : " )
			<< f->getPosFormatted( getObjectLength( i ), isObjectRetrograde(i), DEG_PRECISION_SECOND )
			<< Endl;
	}
	o << Endl;
	for( i = HOUSE1; i <= HOUSE12; i++ )
	{
		o << i+1 << wxT( " " ) << f->getPosFormatted( getHouse( i, vedic, false ), false, DEG_PRECISION_SECOND ) << Endl;
	}
}


