/*

	This is the source code of

   	M A I T R E Y A
    ===============

	Open source platform for Vedic and western astrology

  File           Jaimini.cpp
  Release        4.1
  Author         Martin Pettau
  Copyright (C)  2003-2006 by the author

  Released under the Artistic License as published by the
  Free Software Foundation, read the file 'COPYING' for more information.

*/

#ifdef __GNUG__
	#pragma implementation "Jaimini.h"
#endif

#include "Jaimini.h"

#include <wx/string.h>

#include "Conf.h"
#include "func.h"
#include "Horoscope.h"
#include "IPlanet.h"
#include "Lang.h"
#include "Varga.h"
#include "Writer.h"

extern Config *config;

/*****************************************************
**
**   JaiminiExpert   ---   update 
**
******************************************************/
void JaiminiExpert::update()
{
	int i;

	for( i = 0; i < 12; i++ ) nbplanets[i] = 0;
	for( i = ISUN; i <= IKETU; i++ ) { nbplanets[view->getRasi(i)]++; }

	aspSat = calcPlanetAspects( ISATURN );
	aspRah = calcPlanetAspects( IRAHU );
	aspMar = calcPlanetAspects( IMARS );
	aspKet = calcPlanetAspects( IKETU );
	
	calcNodeStrength();

	for( i = 0; i < 12; i++ )
	{
		nbAspectsOnRasi[i] = calcSignAspects( i );
	}
	calcYears();
	calcCharaKarakas();

}

/*****************************************************
**
**   JaiminiExpert   ---   calcPada 
**
******************************************************/
int JaiminiExpert::calcPada( const int &bhava )
{
	assert(( bhava < 12 ) && ( bhava >= 0 ));

	int rasi = red12( view->getRasi( IASCENDANT ) + bhava );
	int lrd = getJaiminiLord( rasi );
	int diff = red12( view->getRasi( lrd ) - rasi );
	int pada = red12( rasi + 2 * diff );
	if ( ! config->iArudhaMode && ! ( diff % 3 )) pada = red12( pada + 9 );
	return pada;
}


/*****************************************************
**
**   JaiminiExpert   ---   getStrongerRasi 
**
******************************************************/
int JaiminiExpert::getStrongerRasi( const int &rasi1, const int &rasi2 )
{
	// RULES IMPLEMENTIEREN !!

	// Rule 1a
	if ( nbplanets[rasi1] > nbplanets[rasi2] )
	{
		return rasi1;
	}
	// Rule 1b
	else if ( nbplanets[rasi1] < nbplanets[rasi2] )
	{
		return rasi2;
	}
	// Rule 2a
	else if ( nbAspectsOnRasi[rasi1] > nbAspectsOnRasi[rasi2] )
	{
		return rasi1;
	}
	// Rule 2b
	else if ( nbAspectsOnRasi[rasi1] < nbAspectsOnRasi[rasi2] )
	{
		return rasi2;
	}
	// **** TODO RULE3
	// Rule 4a
	else if ( view->getRasiLength(getJaiminiLord( rasi1 )) > view->getRasiLength(getJaiminiLord(rasi2 )) )
	{
		return rasi1;
	}
	// Rule 4b
	else if ( view->getRasiLength(getJaiminiLord( rasi1 )) < view->getRasiLength(getJaiminiLord(rasi2 )) )
	{
		return rasi2;
	}
	else assert( false );
	return 0;
}

/*****************************************************
**
**   JaiminiExpert   ---   getJaiminiLord 
**
******************************************************/
int JaiminiExpert::getJaiminiLord( const int &rasi )
{
	int lrd = ::getLord( rasi );
	if (( rasi == SCORPIO ) && ( ketuStrongerThanMars )) lrd = IKETU;
	if (( rasi == AQUARIUS ) && ( rahuStrongerThanSaturn )) lrd = IRAHU;

	return lrd;
}

/*****************************************************
**
**   JaiminiExpert   ---   isKetuStrongerThanMars 
**
******************************************************/
bool JaiminiExpert::isKetuStrongerThanMars()
{
	return ketuStrongerThanMars;
}

/*****************************************************
**
**   JaiminiExpert   ---   isRahuStrongerThanSaturn 
**
******************************************************/
bool JaiminiExpert::isRahuStrongerThanSaturn()
{
	return rahuStrongerThanSaturn;
}

/*****************************************************
**
**   JaiminiExpert   ---   write
**
******************************************************/
void JaiminiExpert::write( Writer *writer )
{
	int i;
	Lang *lang = Lang::get();	
	VargaExpert expert;
	wxString s;

	writer->writeHeader1 ( _( "Jaimini" ));
	s.Printf( wxT( "%s: %s"), _( "Varga"), expert.getVargaName( view->getDivision()) );
	writer->writeLine( s );

	writer->writeHeader2( _( "Chara Karakas" ));
	Table t1( 2, 9 );
	t1.setHeader( 0, _( "Karaka" ));
	t1.setHeader( 1, _( "Planet" ));
	int line = 1;
	for( i = ATMA_KARAKA; i <= DARA_KARAKA; i++ )
	{
		t1.setEntry( 0, line, lang->getKarakaName( i ) );
		t1.setEntry( 1, line, lang->getObjectName( getCharaKaraka(i), TLARGE ));
		line++;
	}
	writer->writeTable( t1 );

	writer->writeHeader2( _( "Stronger Planets" ));
	Table t2( 2, 3 );
	t2.setHeader( 0, _( "Pair" ));
	t2.setHeader( 1, _( "Stronger" ));
	t2.setEntry( 0, 1, _( "Rahu/Saturn" ));
	t2.setEntry( 0, 2, _( "Ketu/Mars" ));
	t2.setEntry( 1, 1, ( isRahuStrongerThanSaturn() ? lang->getObjectName( IRAHU, TLARGE )
		: lang->getObjectName( ISATURN, TLARGE )) );

	t2.setEntry( 1, 2, ( isKetuStrongerThanMars() ? lang->getObjectName( IKETU, TLARGE )
		: lang->getObjectName( IMARS, TLARGE )));
	writer->writeTable( t2 );

	writer->writeHeader2( _( "Arudha Padas" ));
	Table t3( 2, 13 );
	t3.setHeader( 0, _( "House" ));
	t3.setHeader( 1, _( "Rasi" ));
	for( i = 0; i < 12; i++ )
	{
		s.Printf( wxT( "%02d " ), i+1 );
		t3.setEntry( 0, i+1, s );
		t3.setEntry( 1, i+1, lang->getSignName( calcPada( i ), TLARGE ) );
	}
	writer->writeTable( t3 );

	writer->writeHeader2( _( "Years" ));
	Table t4( 2, 13 );
	t4.setHeader( 0, _( "Rasi" ));
	t4.setHeader( 1, _( "Years" ));
	for( i = 0; i < 12; i++ )
	{
		t4.setEntry( 0, i+1, lang->getSignName( i, TLARGE ));
		s.Printf( wxT( "%02d" ), getRasiYears( i ) );
		t4.setEntry( 1, i+1, s );
	}
	writer->writeTable( t4 );
}

/*****************************************************
**
**   JaiminiExpert   ---   calcPlanetAspects 
**
******************************************************/
int JaiminiExpert::calcPlanetAspects( const int &planet )
{
	int ret = 0;

	if ( hasJaiminiAspect( view->getRasi(IJUPITER), view->getRasi(planet) )) ret++;
	if ( hasJaiminiAspect( view->getRasi(IMERCURY), view->getRasi(planet) )) ret++;

	int rasi1 = view->getRasi(planet);
	int lord1 = getJaiminiLord( rasi1 );
	if ( hasJaiminiAspect( view->getRasi(lord1), view->getRasi(planet) )) ret++;

	return ret;
}


/*****************************************************
**
**   JaiminiExpert   ---   calcNodeStrength 
**
******************************************************/
void JaiminiExpert::calcNodeStrength()
{
	ketuStrongerThanMars = false;
	rahuStrongerThanSaturn = false;

	/**********************************************
	** RAHU -- SATURN
	**********************************************/
	// ** RULE 1: Jupiter aspects rahu -> rahu in fall
	if ( hasJaiminiAspect( view->getRasi(IJUPITER), view->getRasi(IRAHU) ))
	{
		rahuStrongerThanSaturn = false;
	}
	// RULE 2a
	else if (( view->getRasi(ISATURN) == AQUARIUS ) && ( view->getRasi(IRAHU) != AQUARIUS ))
	{
		rahuStrongerThanSaturn = true;
	}
	// RULE 2b
	else if (( view->getRasi(ISATURN) != AQUARIUS ) && ( view->getRasi(IRAHU) == AQUARIUS ))
	{
		rahuStrongerThanSaturn = false;
	}
	// RULE 3a
	else if ( nbplanets[view->getRasi(ISATURN)] > nbplanets[view->getRasi(IRAHU)] )
	{
		rahuStrongerThanSaturn = false;
	}
	// RULE 3b
	else if ( nbplanets[view->getRasi(ISATURN)] < nbplanets[view->getRasi(IRAHU)])
	{
		rahuStrongerThanSaturn = true;
	}
	// RULE 4a
	else if ( aspSat > aspRah )
	{
		rahuStrongerThanSaturn = false;
	}
	// RULE 4b
	else if ( aspSat < aspRah )
	{
		rahuStrongerThanSaturn = true;
	}
	// RULE 5a
	else if ( hasExaltationRasi( ISATURN, view->getRasi(ISATURN) ) && ! hasExaltationRasi( IRAHU, view->getRasi(IRAHU) ))
	{
		rahuStrongerThanSaturn = false;
	}
	// RULE 5b
	else if ( ! hasExaltationRasi( ISATURN, view->getRasi(ISATURN) ) && hasExaltationRasi( IRAHU, view->getRasi(IRAHU) ))
	{
		rahuStrongerThanSaturn = true;
	}
	// RULE 8a
	else if ( view->getRasiLength(ISATURN) > view->getRasiLength(IRAHU) )
	{
		rahuStrongerThanSaturn = false;
	}
	// RULE 8b
	else if ( view->getRasiLength(ISATURN) < view->getRasiLength(IRAHU) )
	{
		rahuStrongerThanSaturn = true;
	}
	else
	{
		rahuStrongerThanSaturn = false;
		//exit(1);
	}

	/**********************************************
	** KETU -- MARS
	**********************************************/
	//** RULE 1: Jupiter aspects ketu -> ketu in fall
	if ( hasJaiminiAspect( view->getRasi(IJUPITER), view->getRasi(IKETU) ))
	{
		ketuStrongerThanMars = false;
	}
	// RULE 2a
	else if (( view->getRasi(IMARS) == SCORPIO ) && ( view->getRasi(IKETU) != SCORPIO ))
	{
		ketuStrongerThanMars = true;
	}
	// RULE 2b
	else if (( view->getRasi(IMARS) != SCORPIO ) && ( view->getRasi(IKETU) == SCORPIO ))
	{
		ketuStrongerThanMars = false;
	}
	// RULE 3a
	else if ( nbplanets[view->getRasi(IMARS)] > nbplanets[view->getRasi(IKETU)] )
	{
		ketuStrongerThanMars = false;
	}
	// RULE 3b
	else if ( nbplanets[view->getRasi(IMARS)] < nbplanets[view->getRasi(IKETU)] )
	{
		ketuStrongerThanMars = true;
	}
	// RULE 4a
	else if ( aspMar > aspKet )
	{
		ketuStrongerThanMars = false;
	}
	// RULE 4b
	else if ( aspMar < aspKet )
	{
		ketuStrongerThanMars = true;
	}
	// RULE 5a
	else if ( hasExaltationRasi( IMARS, view->getRasi(IMARS) ) && ! hasExaltationRasi( IKETU, view->getRasi(IKETU) ))
	{
		ketuStrongerThanMars = false;
	}
	// RULE 5b
	else if ( ! hasExaltationRasi( IMARS, view->getRasi(IMARS) ) && hasExaltationRasi( IKETU, view->getRasi(IKETU) ))
	{
		ketuStrongerThanMars = true;
	}
	// **** TODO: RULE 6 + 7
	// RULE 8a
	else if ( view->getRasiLength(IMARS) > view->getRasiLength(IKETU) )
	{
		ketuStrongerThanMars = false;
	}
	// RULE 8b
	else if ( view->getRasiLength(IMARS) < view->getRasiLength(IKETU) )
	{
		ketuStrongerThanMars = true;
	}
	else
	{
		ketuStrongerThanMars = false;
		//exit(1);
	}
}

/*****************************************************
**
**   JaiminiExpert   ---   calcSignAspects 
**
******************************************************/
int JaiminiExpert::calcSignAspects( const int &rasi )
{
	int ret = 0;

	if ( hasJaiminiAspect( view->getRasi(IJUPITER), rasi )) ret++;
	if ( hasJaiminiAspect( view->getRasi(IMERCURY), rasi )) ret++;

	int lrd = getJaiminiLord( rasi );

	if ( hasJaiminiAspect( view->getRasi(lrd), rasi )) ret++;

	return ret;
}


/*****************************************************
**
**   JaiminiExpert   ---   calcYears 
**
******************************************************/
void JaiminiExpert::calcYears()
{
	for( int rasi = 0; rasi < 12; rasi++ )
	{

		int ll = getJaiminiLord( rasi );
		int pos = view->getRasi( ll );

		if ( isOddFootedRasi( rasi ))
		{
			rasiYears[rasi] = red12( pos - rasi );
		}
		else
		{
			rasiYears[rasi] = red12( rasi - pos );
		}
		
		if ( rasiYears[rasi] == 0 )
		{
			rasiYears[rasi] = 12;
		}
		else
		{
			if (( hasExaltationRasi( ll, pos  )) && ( ll != IMERCURY  ))
			{
				rasiYears[rasi]++;
			}
			else if ( hasDebilationRasi( ll, pos  ))
			{
				rasiYears[rasi]--;
			}
		}
	}
}

/*****************************************************
**
**   JaiminiExpert   ---   getRasiYears 
**
******************************************************/
int JaiminiExpert::getRasiYears( const int &rasi )
{
	assert(( rasi >= 0 ) && ( rasi < 12 ));
	return rasiYears[rasi];
} 

/*****************************************************
**
**   JaiminiExpert   ---   getCharaKaraka 
**
******************************************************/
int JaiminiExpert::getCharaKaraka( const int &index )
{
	assert( index >= ATMA_KARAKA && index <= DARA_KARAKA );
	return chara_karaka[index];
}

/*****************************************************
**
**   JaiminiExpert   ---   getCharaKarakaProperty 
**
******************************************************/
int JaiminiExpert::getCharaKarakaProperty( const int &planet )
{
	assert( planet >= ISUN && planet <= IRAHU );
	return p_chara_karaka[planet];
}

/*****************************************************
**
**   JaiminiExpert   ---   calcCharaKarakas 
**
******************************************************/
void JaiminiExpert::calcCharaKarakas()
{
	int i, j;
	double rl[8], max_len;

	for( i = 0; i < 8; i++ )
	{
		rl[i] = ::getRasiLen( view->getHoroscope()->getObjectLength(i), i == IRAHU );
	}
	for( i = 0; i < 8; i++ )
	{
		max_len = rl[0];
		chara_karaka[i] = ISUN;
		for( j = IMOON; j <= IRAHU; j++ )
		{
			if ( rl[j] > max_len )
			{
				max_len = rl[j];
				chara_karaka[i] = j;
			}
		}
		rl[chara_karaka[i]] = 0;
		p_chara_karaka[chara_karaka[i]] = i;
	}
}

