/*

	This is the source code of

   	M A I T R E Y A
    ===============

	Open source platform for Vedic and western astrology

  File           Lang.cpp
  Release        4.1
  Author         Martin Pettau
  Copyright (C)  2003-2006 by the author

  Released under the Artistic License as published by the
  Free Software Foundation, read the file 'COPYING' for more information.

*/

#ifdef __GNUG__
	#pragma implementation "Lang.h"
#endif

#include "Lang.h"

#include <wx/log.h>
#include <wx/string.h>
#include <math.h>
#include <ctype.h>

#include "constants.h"
#include "Conf.h"
#include "func.h"
#include "Session.h"
#include "Calculator.h"

// round degrees about 5% of a second of arc
#define ROUND_DEGREES .005 / 3600

extern Config *config;

Lang *Lang::ego = 0;
/*****************************************************
**
**   Lang   ---   Constructor 
**
******************************************************/
Lang::Lang()
{
	ego = this;
}

/**************************************************************
***
** Lang   ---   get
***
***************************************************************/
Lang *Lang::get()
{
	if ( ! ego ) ego = new Lang();
	return ego;
}

/**************************************************************
***
** Lang   ---   getNakshatraName
***
***************************************************************/
const wxString Lang::getNakshatraName( const int &nak, const int &nsys, const int &format )
{
	assert(( nsys == N27 ) || ( nsys == N28 ));
	assert(( nsys > 0 ) || ( nsys < ( nsys == N27 ? 27 : 28 ) ));

	const wxChar *k_nakshatra_name_large[28]  = { _( "Aswini" ), _( "Bharani" ), _( "Krittika" ), _( "Rohini" ), 
		_( "Mrigasira" ), _( "Ardra" ), _( "Punarvasu" ), _( "Pushyami" ), _( "Aslesha" ), _( "Magha" ),
		_( "Poorvaphalguni" ), _( "Uttaraphalguni" ), _( "Hasta" ), _( "Chitra" ), _( "Swati" ), _( "Visakha" ),
		_( "Anuradha" ), _( "Jyeshta" ), _( "Moola" ), _( "Poorvashadha" ), _( "Uttarashadha" ), _( "Abhijit" ),
		_( "Sravana" ), _( "Dhanista" ), _( "Satabhisha" ), _( "Poorvabhadra" ), _( "Uttarabhadra" ), _( "Revati" ) };
	const wxChar *k_nakshatra_name_medium[28]  = {
		_( "Aswini" ), _( "Bharani" ), _( "Krittika" ), _( "Rohini" ), _( "Mrigasira" ), _( "Ardra" ),
		_( "Punarvasu" ), _( "Pushyami" ), _( "Aslesha" ), _( "Magha" ), _( "P.Phalguni" ), _( "U.Phalguni" ),
		_( "Hasta" ), _( "Chitra" ), _( "Swati" ), _( "Visakha" ), _( "Anuradha" ), _( "Jyeshta" ),
		_( "Moola" ), _( "P.Shadha" ), _( "U.Shadha" ), _( "Abhijit" ), _( "Sravana" ), _( "Dhanista" ),
		_( "Satabisha" ), _( "P.Bhadra" ), _( "U.Bhadra" ), _( "Revati" ) };
	const wxChar *k_nakshatra_name_short[28]  = { _( "Asw" ), _( "Bhr" ), _( "Kri" ), _( "Roh" ), 
		_( "Mri" ), _( "Ard" ), _( "Pun" ), _( "Pus" ), _( "Asl" ), _( "Mak" ), _( "PPl" ),
		_( "UPl" ), _( "Hst" ), _( "Cit" ), _( "Sva" ), _( "Vis" ), _( "Anu" ), _( "Jye" ),
		_( "Mul" ), _( "PSa" ), _( "USa" ), _( "Abh" ), _( "Sra" ), _( "Dha" ), _( "Sat" ),
		_( "PBa" ), _( "UBa" ), _( "Rev" ) };
	// shift index after  U. shadha for system of 27 nakshatras
	const int index = (( nsys == N27 ) && ( nak > 20 )) ? nak+1 : nak;

	if ( format == TSHORT ) return k_nakshatra_name_short[index];
	else if ( format == TMEDIUM ) return k_nakshatra_name_medium[index];
	else return k_nakshatra_name_large[index];
}

/**************************************************************
***
** Lang   ---   getObjectName
***
***************************************************************/
const wxString Lang::getObjectName( const int &num, const int &format )
{
	assert( ( num >= 0 ) && ( num < MAX_NAMED_OBJECTS ));

	//static const wxChar *k_object_large[MAX_NAMED_OBJECTS] =
	static const wxChar *k_object_large[MAX_NAMED_OBJECTS] =
	{
		_("Sun "), _("Moon"), _("Mars"), _("Mercury"), _("Jupiter" ), _("Venus"), _("Saturn"),
		_("Rahu"), _("Ketu"), _("Ascendant"),
		_("Bhava Lagna"), _("Hora Lagna"), _("Ghatika Lagna"),
		_("Dhuma"), _("Vyatipata"), _("Parivesha"), _("Chapa"),
		_("Upaketu"), _("Kala"), _("Mrityu"), _("Ardhapra"), _("Yamaghan"), _("Gulika"),
		_("Sun "), _("Moon"), _("Mercury"), _("Venus"), _("Mars"), _("Jupiter"), _("Saturn"),
		_("Uranus"), _("Neptune"), _("Pluto"), _("Lunar Node"), _("Ascendant"), _("Medium Coeli"),
		_("Cupido"), _("Hades"), _("Zeus"), _("Kronos"), _("Apollon"), _("Admetos"), _("Vulkanus"), _("Poseidon"),
		_( "Aries" )
	};
	if ( format == TLARGE ) {	return k_object_large[num]; }
	else if ( format == TMEDIUM ) { return wxString( k_object_large[num] ).Left( 3 ); }
	else { return wxString( k_object_large[num] ).Left( 2 ); }
}

/*****************************************************
**
**   Lang   ---   mapO2WIndex 
**
******************************************************/
int Lang::mapO2WIndex( const int &i )
{
	if ( i >= OSUN && i <= OPLUTO ) { return WSUN + i; }
	else if ( i == OMEANNODE || i == OTRUENODE ) { return WLUNARNODE; }
	else if ( i >= OCUPIDO && i <= OARIES ) { return i - OCUPIDO + WCUPIDO; }
	else if ( i == OASCENDANT ) { return WASCENDANT; }
	else if ( i == OMC ) { return WMC; }
	else assert( 0 );
return -1;
}

/**************************************************************
***
** Lang   ---   getOListObjectName
***
***************************************************************/
const wxString Lang::getOListObjectName( const int &num, const int &format )
{
	assert( num >= 0 && num < 32 );
	return getObjectName( mapO2WIndex( num ), format );
}

/**************************************************************
***
** Lang   ---   getYoniName
***
***************************************************************/
const wxString Lang::getYoniName( const int &index )
{
	assert( index >= 0 && index < 14 );
	/*
	const wxChar *k_yoni_name_sk[14] = { "Ashwa", "Mahish", "Simha", "Gaja", "Mesha", "Vanar",
	"Nakul", "Sarpa", "Mriga", "Swan", "Marjar", "Mushak", "Vyaghra", "Gow" };
	*/
	const wxChar *k_yoni_name[14] = { _( "Horse" ), _( "Buffalo" ), _( "Lion" ), _( "Elephant" ), _( "Goat" ), _( "Monkey" ),
	_( "Mongoose" ), _( "Serpent" ), _( "Deer" ), _( "Dog" ), _( "Cat" ), _( "Rat" ), _( "Tiger" ), _( "Cow"  ) };
	return k_yoni_name[index];
}

/**************************************************************
***
** Lang   ---   getGanaName
***
***************************************************************/
const wxString Lang::getGanaName( const int &index )
{
	assert( index >= 0 && index < 3 );
	const wxChar *k_gana_name[3] = { _( "Deva" ), _( "Manuj" ), _( "Rakshas" ) };
	return k_gana_name[index];
}

/**************************************************************
***
** Lang   ---   getTaraName
***
***************************************************************/
const wxString Lang::getTaraName( const int &index )
{
	assert( index >= 0 && index < 9 );
	const wxChar *k_tara_name[9] = { _( "Janma" ), _( "Sampat" ), _( "Vipat" ), _( "Kshema" ),
    _( "Pratyak" ), _( "Sadhaka" ), _( "Vadha" ), _( "Mitra" ), _( "Parma Mitra" ) };

	return k_tara_name[index];
}

/**************************************************************
***
** Lang   ---   getNadiName
***
***************************************************************/
const wxString Lang::getNadiName( const int &index )
{
	assert( index >= 0 && index < 3 );
	const wxChar *k_nadi_name[3] = { _( "Adi (Vata)" ), _( "Madhya (Pitta)" ), _( "Antya (Kapha)" ) };
	return k_nadi_name[index];
}

/**************************************************************
***
** Lang   ---   getVarnaName
***
***************************************************************/
const wxString Lang::getVarnaName( const int &index )
{
	assert( index >= 0 && index < 4 );
	const wxChar *k_varna_name[4] = { _( "Shudra" ), _("Vaishya" ), _( "Kshattriya" ), _( "Brahmin" ) };
	return k_varna_name[index];
}

/**************************************************************
***
** Lang   ---   getTithiName
***
***************************************************************/
const wxString Lang::getTithiName( const int &index )
{
	assert( index >= 0 && index < 30 );
	const wxChar *k_tithi_name[30] = {
	_( "Sukla - Pratipad" ),
	_( "Sukla - Dviteeya" ),
	_( "Sukla - Triteeya" ),
	_( "Sukla - Chaturthi" ),
	_( "Sukla - Panchami" ),
	_( "Sukla - Shasti" ),
	_( "Sukla - Saptami" ),
	_( "Sukla - Asthami" ),
	_( "Sukla - Navami" ),
	_( "Sukla - Dasami" ),
	_( "Sukla - Ekadasi" ),
	_( "Sukla - Dvadasi" ),
	_( "Sukla - Trayodasi" ),
	_( "Sukla - Chaturdasi" ),
	_( "Sukla - Purnima" ),
	_( "Krishna - Pratipad" ),
	_( "Krishna - Dviteeya" ),
	_( "Krishna - Triteeya" ),
	_( "Krishna - Chaturthi" ),
	_( "Krishna - Panchami" ),
	_( "Krishna - Shasti" ),
	_( "Krishna - Saptami" ),
	_( "Krishna - Asthami" ),
	_( "Krishna - Navami" ),
	_( "Krishna - Dasami" ),
	_( "Krishna - Ekadasi" ),
	_( "Krishna - Dvadasi" ),
	_( "Krishna - Trayodasi" ),
	_( "Krishna - Chaturdasi" ),
	_( "Krishna - Amavasya" ),
	};
	return k_tithi_name[index];
}

/**************************************************************
***
** Lang   ---   getWeekdayName
***
***************************************************************/
const wxString Lang::getWeekdayName( const int &day )
{
	assert(( day >= 0 ) || ( day < 7 ));

	static const wxChar *wd[7] = { _("Sunday"), _("Monday"), _("Tuesday"), _("Wednesday"),
		_("Thursday"), _("Friday"), _("Saturday") };
	return wd[day];
}

/**************************************************************
***
** Lang   ---   getSignSymbolCode
***
***************************************************************/
/*
const wxChar Lang::getSignSymbolCode( const int &i )
{
	return (wxChar)( i + 'a' );
}
*/

/**************************************************************
***
** Lang   ---   getPlanetSymbolCode
***
***************************************************************/
/*
const wxChar Lang::getPlanetSymbolCode( const int &i )
{
	return (wxChar)( i + 'A' - WSUN );
}
*/

/**************************************************************
***
** Lang   ---   getSignName
***
***************************************************************/
const wxString Lang::getSignName( const int &i, const int format )
{
	assert(( i >= 0 ) || ( i < 12 ));
	wxString s;

	const wxChar *k_sign[12]={ _( "Ar" ), _( "Ta" ), _( "Ge" ), _( "Cn" ), _( "Le" ), _( "Vi" ),
		_( "Li" ), _( "Sc" ), _( "Sa" ), _( "Cp" ), _( "Aq" ), _( "Pi" ) };
	const wxChar *k_sign_full[12]={ _( "Aries" ), _( "Taurus" ), _( "Gemini" ), _( "Cancer" ), _( "Leo" ),
		_( "Virgo" ), _( "Libra" ), _( "Scorpio" ), _( "Sagittarius" ), _( "Capricorn" ), _( "Aquarius" ), _( "Pisces" )};

	/* Problem: Vri ist not unique
	const wxChar *k_sign_vedic[12]={ wxT( "Ms" ), wxT( "Vs" ), wxT( "Mi" ), wxT( "Kt" ), wxT( "Si" ), wxT( "Kn" ),
		wxT( "Tu" ), wxT( "Vc" ), wxT( "Da" ), wxT( "Mk" ), wxT( "Ku" ), wxT( "Mn" ) };
	const wxChar *k_sign_full_vedic[12]={ wxT( "Mesha" ), wxT( "Vrishabha" ), wxT( "Mithuna" ), wxT( "Kataka" ), wxT( "Simha" ),
		wxT( "Kanya" ), wxT( "Thula" ), wxT( "Vrischika" ), wxT( "Dhanus" ), wxT( "Makara" ), wxT( "Kumbha" ), wxT( "Meena" )};
		*/

	if ( format == TLARGE )
	{
		// s = ( config->useVedicSigns ? k_sign_full_vedic[i] :  k_sign_full[i] );
		s = k_sign_full[i];
	}
	else if ( format == TMEDIUM )
	{
		//s = ( config->useVedicSigns ? k_sign_full_vedic[i] :  k_sign_full[i] );
		s = k_sign_full[i] ;
		s = s.Left( 3 );
	}
	else {
		//s = ( config->useVedicSigns ? k_sign_vedic[i] :  k_sign[i] );
		s = k_sign[i];
	}
	return s;
}

/**************************************************************
***
** Lang   ---   getKarakaName
***
***************************************************************/
const wxString Lang::getKarakaName( const int &i, const int format )
{
	const wxChar *chara_karaka_name_long[8] = { _( "Atma" ), _( "Amatya" ), _( "Bhratru" ),
		_( "Matru" ), _( "Pitru" ), _( "Putra" ), _( "Gnati" ), _( "Dhanya"  ) };
	assert( i >= 0 && i < 8 );
	return chara_karaka_name_long[i];
}

/**************************************************************
***
**    Lang   ---   getShastiamsaName
***
***************************************************************/
const wxString Lang::getShastiamsaName( const int &i )
{
	const wxChar *k_shastiamsa_name[60] = {
		_( "Ghora" ), _( "Rakshasa" ), _( "Deva" ), _( "Kubera" ), _( "Yaksha" ),
		_( "Kinnara" ), _( "Bhrashta" ), _( "Kulaghna" ), _( "Garala" ), _( "Vahni" ),
		_( "Maya" ), _( "Purishaka" ), _( "Apampathi" ), _( "Marut" ), _( "Kaala" ),
		_( "Sarpa" ), _( "Amrita" ), _( "Indu" ), _( "Mridu" ), _( "Komala" ),
		_( "Heramba" ), _( "Brahma" ), _( "Vishnu" ), _( "Maheswara" ), _( "Deva" ),
		_( "Ardra" ), _( "Kalinasa" ), _( "Kshiteesa" ), _( "Kamalakara" ), _( "Gulika" ),
		_( "Mrithyu" ), _( "Kaala" ), _( "Davagani" ), _( "Ghora" ), _( "Yama" ),
		_( "Kantaka" ), _( "Sudha" ), _( "Amrita" ), _( "Poornachandra" ), _( "Vishadagdha" ),
		_( "Kulanasa" ), _( "Vamsakshaya" ), _( "Utpata" ), _( "Kaala" ), _( "Saumya" ),
		_( "Komala" ), _( "Seetala" ), _( "Karaladamshtra" ), _( "Chandramukhi" ), _( "Praveena" ),
		_( "Kaala Pavaka" ), _( "Dandayudha" ), _( "Nirmala" ), _( "Saumya" ), _( "Kroora" ),
		_( "Atiseetala" ), _( "Amrita" ), _( "Payodhi" ), _( "Bhramana" ), _( "Chandra Rekha" )
	};
	assert( i >= 0 && i < 60 );
	return k_shastiamsa_name[i];
}

/**************************************************************
***
**    Lang   ---   getBhavaName
***
***************************************************************/
const wxString Lang::getBhavaName( const int &i )
{
	const wxChar *k_bhava_name[12] = { _( "Lagna" ), _( "Dhana" ), _( "Sahaja" ), _( "Bandhu" ),
	_( "Putra" ), _( "Satru" ), _( "Yuvati" ), _( "Randhra" ), _( "Dharma" ), _( "Karma" ),
	_( "Labha" ), _( "Vyaya" ) };
	
	assert( i >= 0 && i < 12 );
	return k_bhava_name[i];
}

/**************************************************************
***
**    Lang   ---   getMonthName
***
***************************************************************/
const wxString Lang::getMonthName( const int &i )
{
	assert( i >= 0 && i < 12 );
	const wxChar *month_name[12] = { _( "January" ), _( "February" ), _( "March" ), _( "April" ),
		_( "May" ), _( "June" ), _( "July" ),
		_( "August" ), _( "September" ), _( "October" ), _( "November" ), _( "December" ) };
	return month_name[i];
}

Formatter *Formatter::ego = 0;
/*****************************************************
**
**   Formatter   ---   Constructor 
**
******************************************************/
Formatter::Formatter()
{
	ego = this;
}

/**************************************************************
***
** Formatter   ---   get
***
***************************************************************/
Formatter *Formatter::get()
{
	if ( ! ego ) ego = new Formatter();
	return ego;
}

/*****************************************************
**
**   Formatter   ---   getDegMinSecInts2 
**
******************************************************/
void Formatter::getDegMinSecInts2( const double &len, int &deg, int &min, double &sec )
{
	double mylen = fabs( len ) + ROUND_DEGREES;
  deg = (int)mylen;
  double mind = (double)((mylen - (double)deg ) * 60);
  min = (int)mind;
  sec = ( mind - (double)min ) * 60;
	if ( len < 0 ) deg *= -1;
}

/*****************************************************
**
**   Formatter   ---   getDegMinSecInts 
**
******************************************************/
void Formatter::getDegMinSecInts( const double &len, int &deg, int &min, int &sec )
{
	double mylen = fabs( len ) + ROUND_DEGREES;
  deg = (int)mylen;
  double mind = (double)((mylen - (double)deg ) * 60);
  min = (int)mind;
  sec = (int)(( mind - (double)min ) * 60 );
	if ( len < 0 ) deg *= -1;
}

/*****************************************************
**
**   Formatter   ---   getLenFormated 
**
******************************************************/
const wxString Formatter::getLenFormated( const double &len, const int &format )
{
	int deg, min, sec;
	wxString s;
	getDegMinSecInts( len, deg, min, sec );

	if (format == DEG_PRECISION_SECOND )
		s.Printf( wxT( "%02d %02d'%02d\"" ), deg, min, sec );
	else
		s.Printf( wxT( "%02d %02d'" ), deg, min );
	return s;
}

/*****************************************************
**
**   Formatter   ---   getTimeFormated 
**
******************************************************/
const wxString Formatter::getTimeFormated( const double &t )
{
	wxString s;
	if (( t > 24 ) || ( t < 0 ))
		wxLogError( wxT( "Warning: time value is %f" ), t );

	int deg, min, sec;
	double mytime = a_red( t, 24 );
	getDegMinSecInts( mytime, deg, min, sec );

  s.Printf( wxT( "%02d:%02d:%02d" ), deg, min, sec );
	return s;
}

/*****************************************************
**
**   Formatter   ---   getDegreesFormated 
**
******************************************************/
const wxString Formatter::getDegreesFormated( const double &t, const int p )
{
	wxString s;
	int deg, min;
	double sec;
	double mytime = a_red( t, 180 );

	if (( t > 360 ) || ( t < 0 )) wxLogError( wxT( "Warning: time value is %f" ), t );
	getDegMinSecInts2( mytime, deg, min, sec );

	if ( p == DEG_PRECISION_MORE )
		s.Printf( wxT( "%02d:%02d:%02d.%02d" ), deg, min, (int)sec, (int)(100 * ( sec - (int)sec )));
	else s.Printf( wxT( "%02d:%02d:%02d" ), deg, min, (int)sec );
	return s;
}

/*****************************************************
**
**   Formatter   ---   getLatitudeFormated 
**
******************************************************/
const wxString Formatter::getLatitudeFormated( const double &l, const int p )
{
	wxString s;
	s.Printf( wxT( "%s %s" ), getDegreesFormated( fabs( l ), p ).c_str(), ( l >= 0 ? _( "North" ) : _( "South" ) ));
	return s;
}

/*****************************************************
**
**   Formatter   ---   getLongitudeFormated 
**
******************************************************/
const wxString Formatter::getLongitudeFormated( const double &l, const int p )
{
	wxString s;
	s.Printf( wxT( "%s %s" ), getDegreesFormated( fabs( l ), p ).c_str(), ( l >= 0 ? _( "East" ) : _( "West" ) ));
	return s;
}

/*****************************************************
**
**   Formatter   ---   getPosFormatted 
**
******************************************************/
wxString Formatter::getPosFormatted( const double &len, const int &dir, const int &format )
{
	int deg, min, sign;
	double sec;
	wxString d = wxT( " " );
	const int pformat = config->signPrecision;
	wxString s;

	double mylen = red_deg( len );
  sign = (int)( mylen / 30 );
	mylen -= 30 * sign;
	
	getDegMinSecInts2( mylen, deg, min, sec );

	// Rounding
	if ( format == DEG_PRECISION_MINUTE && sec > 30 ) min+= 1;
	if ( format == DEG_PRECISION_SECOND && sec - (int)sec > .5 ) sec += .5;

	if ( dir == DIR_DIRECT ) d = wxT( " " );
	else if ( dir == DIR_RETROGRADE ) d = _( "R" );
	else if ( dir == DIR_STATIONARY ) d = wxT( "S" );
	else if ( dir == DIR_DIRECT_SHOW ) d = _( "D" );
	else if ( dir != DIR_NONE )
	{
		wxLogError( wxT( "Error getPosFormatted : value %d not allowed, exiting.." ), dir );
		exit(1);
	}

	// Don't print direction
  if ( dir == -1 )
	{
		if ( format == DEG_PRECISION_SECOND )
		{
			if ( config->useVedicPositions )
				s.Printf( wxT( "%02d-%02d-%02d-%02d" ), sign, deg, min, (int)sec );
			else 
			{
				s.Printf( wxT( "%02d %02d'%02d\" %s" ), deg, min, (int)sec, Lang::get()->getSignName(sign, pformat).c_str() );
			}
		}
		else if ( format == DEG_PRECISION_MORE )
		{
			if ( config->useVedicPositions )
				s.Printf( wxT( "%02d-%02d-%02d-%02d.%05d" ), sign, deg, min, (int)sec, (int)((sec-(int)sec)*100000));
			else 
				s.Printf( wxT( "%02d %02d'%02d.%05d\" %s" ), deg, min, (int)sec, (int)((sec-(int)sec)*100000),
					Lang::get()->getSignName(sign, format ).c_str() );
		}
		else
		{
			if ( config->useVedicPositions )
				s.Printf( wxT( "%02d-%02d-%02d'" ) , sign, deg, min );
			else 
				s.Printf( wxT( "%02d %02d' %s" ) , deg, min, Lang::get()->getSignName(sign, pformat).c_str() );
		}
	}
  else
	{
		if ( format == DEG_PRECISION_SECOND )
		{
			if ( config->useVedicPositions )
				s.Printf( wxT( "%s %02d-%02d-%02d-%02d" ), d.c_str(), sign, deg, min, (int)sec );
			else 
				s.Printf( wxT( "%s %02d %02d'%02d\" %s" ), d.c_str(), deg, min, (int)sec, Lang::get()->getSignName(sign, pformat).c_str() );
		}
		else if ( format == DEG_PRECISION_MORE )
		{
			if ( config->useVedicPositions )
				s.Printf( wxT( "%s %02d-%02d-%02d-%02d.%05d\"" ),
					d.c_str(), sign, deg, min, (int)sec, (int)((sec-(int)sec)*100000));
			else
				s.Printf( wxT( "%s %02d %02d'%02d.%05d\" %s" ),
					d.c_str(), deg, min, (int)sec, (int)((sec-(int)sec)*100000), Lang::get()->getSignName(sign, pformat).c_str() );
		}
		else
		{
			if ( config->useVedicPositions )
				s.Printf( wxT( "%s %02d-%02d-%02d" ), d.c_str(), sign, deg, min );
			else
				s.Printf( wxT( "%s %02d %02d' %s" ), d.c_str(), deg, min, Lang::get()->getSignName(sign, pformat).c_str() );
		}
	}
	return s;
}

/*****************************************************
**
**   Formatter   ---   getLenString 
**
******************************************************/
const wxString Formatter::getLenString( const double &len, const int format, const int dir )
{
	wxString s;
	int deg, min, sec, sign;
	wxChar d = 'D';

	double mylen = red_deg( len );
  sign = (int)( mylen / 30 );
	mylen -= 30 * sign;
	
	getDegMinSecInts( mylen, deg, min, sec );

	if ( dir == 1 ) d = 'R';

  if ( format & FORMAT_SHOW_DIR ) 
	{
		if ( format & FORMAT_SHOW_SECONDS )
			s.Printf( wxT( "%c %02d %s %02d %02d" ), d, deg, Lang::get()->getSignName(sign).c_str(), min, sec );
		else
			s.Printf( wxT( "%c %02d %s %02d" ), d, deg, Lang::get()->getSignName(sign).c_str(), min );
	}
	else
	{
		if ( format & FORMAT_SHOW_SECONDS )
			s.Printf( wxT( "%02d %s %02d %02d" ), deg, Lang::get()->getSignName(sign).c_str(), min, sec );
		else
			s.Printf( wxT( "%02d %s %02d" ), deg, Lang::get()->getSignName(sign).c_str(), min );
	}
	return s;
}

/*****************************************************
**
**   Formatter   ---   getDateIntsFromJD 
**
******************************************************/
void Formatter::getDateIntsFromJD( const double &jd, int &day, int &month, int &year, const int format )
{
	double ut = 0;
	Session *session = Session::get();
	session->getCalculator()->getDateIntsFromJD( jd, year, month, day, ut );
}

/*****************************************************
**
**   Formatter   ---   getFullDateStringFromJD 
**
******************************************************/
const wxString Formatter::getFullDateStringFromJD( const double &jd, const int format )
{
	wxString s;
	int year, month, day;

	getDateIntsFromJD( jd, day, month, year );
	int l = Session::get()->getLocaleCode();
	if ( l == wxLANGUAGE_GERMAN ) // German
		s.Printf( wxT( "%02d.%02d.%04d %s" ), day, month, year, getTimeFormated( getTimeFromJD(jd)).c_str() );
	else 
		s.Printf( wxT( "%04d-%02d-%02d %s" ), year, month, day, getTimeFormated( getTimeFromJD(jd)).c_str() );
	return s;
}

/*****************************************************
**
**   Formatter   ---   getDateStringFromJD 
**
******************************************************/
const wxString Formatter::getDateStringFromJD( const double &jd, const int format )
{
	wxString s;
	int year, month, day;
	getDateIntsFromJD( jd, day, month, year );
	int l = Session::get()->getLocaleCode();
	if ( l == wxLANGUAGE_GERMAN ) // German
		s.Printf( wxT( "%02d.%02d.%04d" ), day, month, year );
	else
		s.Printf( wxT( "%04d-%02d-%02d" ), year, month, day );
	return s;
}


/*****************************************************
**
**   Formatter   ---   getDateIntsFromString 
**
******************************************************/
bool Formatter::getDateIntsFromString( const wxChar *value, int &day, int &month, int &year, int format )
{
	if ( ! value )
	{
		wxLogError( wxT( "Error: date value is NULL" ));
		return false;
	}
	wxChar *c = (wxChar*)value;
	while( *c )
	{
		if ( ! ( isdigit( *c ) || *c == '.' || *c == '-' )) return false;
		c++;
	}
	int ret;
	int l = Session::get()->getLocaleCode();
	if ( l == wxLANGUAGE_GERMAN ) // German
		ret = wxSscanf( value, wxT( "%d.%d.%d" ), &day, &month, &year );
	else
		ret = wxSscanf( value, wxT( "%d-%d-%d" ), &year, &month, &day );
	if ( ret != 3 ) return false;

	if (( day < 0 ) || ( day > 31 )) return false;
	if (( month < 0 ) || ( month > 12 )) return false;
	if (( year < -3000 ) || ( year > 6000 )) return false;

	return true;
}

/*****************************************************
**
**   Formatter   ---   getDegreeIntsFromString 
**
******************************************************/
bool Formatter::getDegreeIntsFromString( const wxChar *value, int &deg, int &minute, int &second, int format )
{
	if ( ! value )
	{
		wxLogError( wxT( "Error: degree value is NULL" ));
		return false;
	}
	wxChar *c = (wxChar*)value;
	// TODO
	while( *c )
	{
		if ( ! ( isdigit( *c ) || *c == ':' )) return false;
		c++;
	}
	
	deg = 0;
	minute = 0;
	second = 0;

	int ret = wxSscanf( value, wxT( "%d:%d:%d" ), &deg, &minute, &second );
	if ( ret < 1 ) return false;

	if (( deg < -180 ) || ( deg > 180 )) return false;
	if (( minute < 0 ) || ( minute > 59 )) return false;
	if (( second < 0 ) || ( second > 59 )) return false;

	return true;
}

/*****************************************************
**
**   Formatter   ---   getAyanamsaName 
**
******************************************************/
const wxString Formatter::getAyanamsaName( const int &ayatype )
{
	wxString s;
	switch( ayatype )
	{
		case 0:
			s= _( "None" );
		break;
		case 1:
			s = _( "Lahiri" );
		break;
		case 2:
			s = _( "Raman" );
		break;
		case 3:
			s = _( "Krishnamurti" );
		break;
		case 4:
			s.Printf( wxT( "Custom (t0 %8.1f ayan_t0 %2.8f)" ), config->custom_t0, config->custom_ayan_t0 );
		break;
		default:
			s= wxT( "Unknown" );
		break;
	}
	return s;
}




