/*

	This is the source code of

   	M A I T R E Y A
    ===============

	Open source platform for Vedic and western astrology

  File           Midpoint.cpp
  Release        4.1
  Author         Martin Pettau
  Copyright (C)  2003-2006 by the author

  Released under the Artistic License as published by the
  Free Software Foundation, read the file 'COPYING' for more information.

*/

#ifdef __GNUG__
	#pragma implementation "Midpoint.h"
#endif

#include "Midpoint.h"

#include <wx/string.h>
#include <list>
#include <math.h>
#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>

#include "constants.h"
#include "func.h"
#include "Calculator.h"
#include "Conf.h"
#include "DataSet.h"
#include "Horoscope.h"
#include "Lang.h"
#include "Session.h"
#include "Writer.h"

using namespace std;
extern Config *config;

/*****************************************************
**
**   MidpointEvent   ---   Constructor 
**
******************************************************/
MidpointEvent::MidpointEvent( int p, int i0, int h1, int i1, int h2, int i2, double orbis, int type, double jd )
{
	this->p = p;
	this->i0 = i0;
	this->h1 = h1;
	this->i1 = i1;
	this->h2 = h2;
	this->i2 = i2;
	this->orbis = orbis;
	this->type = type;
	this->jd = jd;
}

/*****************************************************
**
**   MidpointEvent   ---   write
**
******************************************************/
void MidpointEvent::write( Writer *writer, wxString &o, bool show_datum )
{
	Lang *lang = Lang::get();
	Formatter *formatter = Formatter::get();

	o << lang->getOListObjectName( p, TSHORT ) << MidpointExpert::getEventSymbol(writer, type)
		<< lang->getOListObjectName( h1, TSHORT )
		<< wxT( "/" ) << lang->getOListObjectName( h2, TSHORT ) << wxT( " " )
		<< ( (!show_datum && orbis < 0) ? wxT( "-" ) : wxT( " " ));
	if ( show_datum ) o << formatter->getDateStringFromJD( jd );
	else o << formatter->getDegreesFormated( fabs( orbis ));

}

/*****************************************************
**
**   ConjunctionEvent   ---   Constructor 
**
******************************************************/
ConjunctionEvent::ConjunctionEvent( const int &p1, const int &i1, const int &p2, const int &i2, const double &orbis,
	const int &type, const double jd )
{
	this->p1 = p1;
	this->i1 = i1;
	this->p2 = p2;
	this->i2 = i2;
	this->orbis = orbis;
	this->type = type;
	this->jd = jd;
}

/*****************************************************
**
**   ConjunctionEvent   ---   write
**
******************************************************/
void ConjunctionEvent::write( Writer *writer, wxString &o, bool show_datum )
{
	Lang *lang = Lang::get();
	Formatter *formatter = Formatter::get();

	o << lang->getOListObjectName( p1, TSHORT ) << MidpointExpert::getEventSymbol(writer, type)
		<< lang->getOListObjectName( p2, TSHORT )
		<< wxT( " " ) << ( !show_datum &&orbis > 0 ? wxT( "-" ) : wxT( " " ));
	if ( show_datum ) o << formatter->getDateStringFromJD( jd );
	else o << formatter->getDegreesFormated( fabs( orbis ));
}

/*****************************************************
**
**   CLASS MidPointSort  
**
******************************************************/
class MidPointSort
{
public:
	MidPointSort( const int &ord ) { order = ord; }
	bool operator()( const MidpointEvent &e1, const MidpointEvent &e2 ) const
	{
		switch( order )
		{
			case MIDPOINT_SORT_PLANET2:
				// erst nach der MidpointValue (Planet2+3), dann nach dem ersten Planeten
				if ( e1.i1 != e2.i1 ) return ( e1.i1 < e2.i1 );
				if ( e1.i2 != e2.i2 ) return ( e1.i2 < e2.i2 );
				return ( e1.p < e2.p );
			break;
			case MIDPOINT_SORT_ORBIS:
				return( e1.orbis < e2.orbis );
			break;
			case MIDPOINT_SORT_ORBIS_BETRAG:
				return( fabs( e1.orbis ) < fabs( e2.orbis ));
			break;
			case MIDPOINT_SORT_PLANET1:
				// erst nach dem 1. Planeten, dann nach den Planeten in der MidpointValue
				if ( e1.i0 != e2.i0 ) return ( e1.i0 < e2.i0 );
				if ( e1.i1 != e2.i1 ) return ( e1.i1 < e2.i1 );
				return ( e1.i2 < e2.i2 );
			break;
			case MIDPOINT_SORT_ASPECT:
				if ( e1.type != e2.type ) return( e1.type < e2.type );
				return( fabs( e1.orbis ) < fabs( e2.orbis ));
			break;
			case MIDPOINT_SORT_DATUM:
				return( e1.jd < e2.jd );
			break;
			case MIDPOINT_SORT_ORBIS_REV:
				return( e1.orbis > e2.orbis );
			break;
			case MIDPOINT_SORT_JD:
				return( e1.jd < e2.jd );
			break;
			default:
				assert( false );
			break;
		}
		return false;
	}
private:
	int order;
};

/*****************************************************
**
**   CLASS KonjSort  
**
******************************************************/
class KonjSort
{
public:
	KonjSort( const int &ord ) { order = ord; }
	bool operator()( const ConjunctionEvent &e1, const ConjunctionEvent &e2 ) const
	{
		switch( order )
		{
			case MIDPOINT_SORT_PLANET2:
				return ( e1.i1 < e2.i1 );
			break;
			case MIDPOINT_SORT_ORBIS:
				return( e1.orbis < e2.orbis );
			break;
			case MIDPOINT_SORT_ORBIS_BETRAG:
				return( fabs( e1.orbis ) < fabs( e2.orbis ));
			break;
			case MIDPOINT_SORT_PLANET1:
				return ( e1.i2 < e2.i2 );
			break;
			case MIDPOINT_SORT_ASPECT:
				if ( e1.type != e2.type ) return( e1.type < e2.type );
				return( fabs( e1.orbis ) < fabs( e2.orbis ));
			break;
			case MIDPOINT_SORT_DATUM:
				return( e1.jd < e2.jd );
			break;
			case MIDPOINT_SORT_ORBIS_REV:
				return( e1.orbis > e2.orbis );
			break;
			case MIDPOINT_SORT_JD:
				return( e1.jd < e2.jd );
			break;
			default:
				assert( false );
			break;
		}
		return false;
	}
private:
	int order;
};

/*****************************************************
**
**   MidpointExpert   ---   Constructor 
**
******************************************************/
MidpointExpert::MidpointExpert( int hsSystem )
{
	this->hsSystem = hsSystem;
	updatePList();
}

/*****************************************************
**
**   MidpointExpert   ---   updateTextEntries 
**
******************************************************/
void MidpointExpert::updateTextEntries()
{
	text_entries.clear();
}

/*****************************************************
**
**   MidpointExpert   ---   getEventSymbol 
**
******************************************************/
const wxChar *MidpointExpert::getEventSymbol( Writer *writer, const int &i )
{
	static const wxChar *event_symbol[8] = { wxT( "*" ), wxT( "=" ), wxT( "|" ), wxT( "#" ), wxT( "<" ),
		wxT( "-" ), wxT( "~" ), wxT( "%" ) };
	static const wxChar *event_symbol_html[8] = { wxT( "*" ), wxT( "=" ), wxT( "|" ), wxT( "#" ),
		wxT( "&lt;" ), wxT( "-" ), wxT( "~" ), wxT( "%" ) };
	assert( i >= 0 && i < 8 );
	if ( writer->type == WRITER_HTML )
		return ( config->hsUseSpecialEventSymbols ? event_symbol_html[i] : wxT( "=" ));
	else
		return ( config->hsUseSpecialEventSymbols ? event_symbol[i] : wxT( "=" ));
}

/*****************************************************
**
**   MidpointExpert   ---   updatePList 
**
******************************************************/
void MidpointExpert::updatePList()
{
	int i, node;
	node = config->wLunarNodeMode == LUNAR_NODE_MEAN ? OMEANNODE : OTRUENODE;

	plist.clear();
	if ( hsSystem )
	{ // Witte
		aspos = 3;
		mcpos = 0;
		sunpos = 2;
		plist.push_back( OMC );
		plist.push_back( OARIES );
		plist.push_back( OSUN );
		plist.push_back( OASCENDANT );
		plist.push_back( OMOON );
		plist.push_back( node );
		for( i = OMERCURY; i <= OPLUTO; i++ ) plist.push_back( i );
		for( i = OCUPIDO; i <= OPOSEIDON; i++ ) plist.push_back( i );
	}
	else
	{ // Ebertin
		aspos = 11;
		mcpos = 12;
		sunpos = 0;
		for( i = OSUN; i <= OPLUTO; i++ ) plist.push_back( i );
		plist.push_back( node );
		plist.push_back( OASCENDANT );
		plist.push_back( OMC );
	}
}

/*****************************************************
**
**   MidpointExpert   ---   updatePos 
**
******************************************************/
void MidpointExpert::updatePos( const int &which, DataSet *d )
{
	unsigned int i;
	double dummy;
	Calculator *calculator = Session::get()->getCalculator();
	Location *loc = d->getLocation();

	double *p = which == 1 ? pos1 : pos2;

	for( i = 0; i < plist.size(); i++ )
	{
		if ( plist[i] >= OSUN && plist[i] <=  OPOSEIDON )
		{
			calculator->calcPosition( d, plist[i], p[i], dummy, true, false );
		}
		else if ( plist[i] == OARIES ) { p[i] = 0; }
		else if ( plist[i] == OASCENDANT ) { p[i] = calculator->calcAscendantAya( d->getJD(), loc->getLatitude(), loc->getLongitude(), false ); }
		else if ( plist[i] == OMC ) { p[i] = calculator->calcMCAya( d->getJD(), loc->getLatitude(), loc->getLongitude(), false ); }
		else assert( 0 );
	}
}

/*****************************************************
**
**   MidpointExpert   ---   mapOToENumber 
**
******************************************************/
int MidpointExpert::mapOToENumber( const int &p )
{
	if ( p >= OSUN && p <= OPLUTO ) return p - OSUN;
	else if ( p == OMEANNODE || p == OTRUENODE ) return 10;
	else if ( p == OASCENDANT ) return 11;
	else if ( p == OMC ) return 12;
	else assert( false );

	return -1;
}

/*****************************************************
**
**   MidpointExpert   ---   mapOToWNumber 
**
******************************************************/
int MidpointExpert::mapOToWNumber( const int &p )
{
	if ( p == -1 ) return -1;
	else if ( p == OMC ) return 0;
	else if ( p == OARIES ) return 1;
	else if ( p == OSUN ) return 2;
	else if ( p == OASCENDANT ) return 3;
	else if ( p == OMOON ) return 4;
	else if ( p == OMEANNODE || p == OTRUENODE ) return 5;
	if ( p >= OMERCURY && p <= OPLUTO ) return p + 4;
	if ( p >= OCUPIDO && p <= OPOSEIDON ) return p + 2;
	else assert( false );

	return -1;
}

/*****************************************************
**
**   MidpointExpert   ---   getPlanetaryIndexW 
**
******************************************************/
void MidpointExpert::getPlanetaryIndexW( char *buf, const int &p, const int &h1, const int &h2 )
{
	const char *codename[22] = {  "mc" ,  "ar" , "so" ,  "as" ,  "mo" , "kn" ,
	"me" ,  "ve" ,  "ma" ,  "ju" ,  "sa" , "ur" , "ne" , "pl" ,
	"cu" , "ha" ,  "ze" ,  "kr" ,  "ap" ,  "ad" ,  "vu" ,  "po"  };
	int q = mapOToWNumber( p );
	int i1 = mapOToWNumber( h1 );
	int i2 = mapOToWNumber( h2);
	assert( q >= -1 && q <= 21 );
	assert( i1 >= 0 && i1 <= 21 );
	assert( i2 >= 0 && i2 <= 21 );
	if ( q >= 0 ) sprintf( buf, "%s%s%s" , codename[q], codename[i1], codename[i2] );
	else sprintf( buf, "%s%s" , codename[Min( i1, i2 )], codename[Max( i1, i2 )] );
}

/*****************************************************
**
**   MidpointExpert   ---   mapPlanetsToNumberE 
**
******************************************************/
int MidpointExpert::mapPlanetsToNumberE( const int &p, const int &h1, const int &h2 )
{
	int q = mapOToENumber( p );
	int i1 = mapOToENumber( h1 );
	int i2 = mapOToENumber( h2);

	int ind = 182;
	for( int i = 0; i < 13; i++ )
	{
		for( int j = i+1; j < 13; j++ )
		{
			ind++;
			for( int k = 0; k < 13; k++ )
			{
				if ( k == j || k == i ) continue;
				// Verdrehen wegen MC <-> AS
				if (( k == q && i == i1 && j == i2 ) || ( k == q && j == i1 && i == i2 )) return ind;
				ind++;
			}
		}
	}
	assert( false );
	return -1;
}

wxChar *gradkreis_name[7] = { wxT( "360" ), wxT( "180" ), wxT( "90" ), wxT( "45" ), wxT( "22.5" ), wxT( "11.25" ), 
	wxT( "5.625" ) };

/*****************************************************
**
**   MidpointExpert   ---   writeRadixAnalysis 
**
******************************************************/
void MidpointExpert::writeRadixAnalysis( Writer *writer, Horoscope *h, const double &orbis,
	const int gradkreis, const int order, const int filter )
{
	wxString s;
  writer->writeHeader1( _( "Midpoint Analysis" ));
	s << _( "Circle" ) << wxT( ": " ) << gradkreis_name[gradkreis] << wxT( " " ) << _( "deg" );
	writer->writeLine( s );
	s.Clear();
	s	<< _( "Orbis" ) << wxT( ": " ) << orbis << wxT( " " ) << _( "min" );
	writer->writeParagraph( s );
	updatePos( 1, h->getDataSet() );
	writePlanetaryPositions( writer, 1, gradkreis ); 
	writeMidpoints( writer, 1, gradkreis );
	calculateEvents( 1, 1, orbis, gradkreis, order, filter );
	writeConjunctions( writer );
	writeMidpointEvents( writer );
	writeMidpointDiagram( writer, 1, 1, orbis, gradkreis, filter );
	writeMidpointEventsText( writer );
}

/*****************************************************
**
**   MidpointExpert   ---   writeMidpoints 
**
******************************************************/
void MidpointExpert::writeMidpoints( Writer *writer, const int &which, const int gradkreis )
{
	int i, j;
	Formatter *formatter = Formatter::get();
	Lang *lang = Lang::get();

	int max1 = ( hsSystem ? 14 : 13); // Ebertin = 0, Witte = 1

	double kreismod = getKreisgrad( gradkreis );
	if ( kreismod > 90 ) kreismod = 90;

  writer->writeHeader2( _( "Midpoint Table" ));
	Table table( max1+2, (int)plist.size()+2 );
	double *p = which == 1 ? pos1 : pos2;

	table.setHeader( 0,  wxEmptyString );
	for( i = 0; i < max1; i++ ) table.setHeader( i+1, lang->getOListObjectName( plist[i], TSHORT ));
	table.setHeader( max1+1, wxEmptyString );
	for( i = 0; i < (int)plist.size(); i++ )
	{
		table.setHeaderEntry( 0, i+1, lang->getOListObjectName( plist[i], TSHORT ));
		for( j = 0; j < max1; j++ )
		{
			if ( i >= j )
			{
				table.setEntry( j+1, i+1, formatter->getDegreesFormated( a_red( ( p[i] + p[j] ) * .5, kreismod )) );
			}
		}
		table.setHeaderEntry( max1+1, i+1, lang->getOListObjectName( plist[i], TSHORT ));
	}
	table.setHeaderEntry( 0, plist.size()+1,  wxEmptyString );
	for( i = 0; i < max1; i++ ) table.setHeaderEntry( i+1, plist.size()+1, lang->getOListObjectName( plist[i], TSHORT ));
	table.setHeaderEntry( max1+1, plist.size()+1, wxEmptyString );
	writer->writeTable( table ); 

	if ( hsSystem ) // Witte
	{
		Table t2( 10, 10 );
		t2.setHeader( 0,  wxEmptyString );
		for( i = max1; i < (int)plist.size(); i++ )
			t2.setHeader( i-max1+1, lang->getOListObjectName( plist[i], TSHORT ));
		t2.setHeader( 9, wxEmptyString );
		for( i = max1; i < (int)plist.size(); i++ )
		{
			t2.setHeaderEntry( 0, i-max1+1, lang->getOListObjectName( plist[i], TSHORT ));
			for( j = max1; j < (int)plist.size(); j++ )
			{
				if ( i >= j )
				{
					t2.setEntry( j-max1+1, i-max1+1, formatter->getDegreesFormated( a_red( ( p[i] + p[j] ) * .5, kreismod )));
				}
			}
			t2.setHeaderEntry( (int)plist.size()-max1+1, i-max1+1, lang->getOListObjectName( plist[i], TSHORT ));
		}
		t2.setHeaderEntry( 0, 9,  wxEmptyString );
		for( i = max1; i < (int)plist.size(); i++ )
			t2.setHeaderEntry( i-max1+1, 9, lang->getOListObjectName( plist[i], TSHORT ));
		//for( i = 0; i < 9; i++ ) t2.setHeaderEntry( i+1, 9, lang->getOListObjectName( plist[i+OCUPIDO], TSHORT ));
		t2.setHeaderEntry( 9, 9, wxEmptyString );
		writer->writeTable( t2 ); 
	}
}

/*****************************************************
**
**   MidpointExpert   ---   writePlanetaryPositions 
**
******************************************************/
void MidpointExpert::writePlanetaryPositions( Writer *writer, const int &which, const int &gradkreis )
{
	unsigned int i;
	double len;
	Formatter *formatter = Formatter::get();
	Lang *lang = Lang::get();

	double *p = which == 1 ? pos1 : pos2;
	int colnr = 4;
	if ( gradkreis > GRADKREIS_45 ) colnr++;
	if ( gradkreis > GRADKREIS_225 ) colnr++;
	Table table( colnr, plist.size()+1 );
	table.setHeader( 0,  _( "Planet" ));
	table.setHeader( 1,  _( "Radix" ));
	table.setHeader( 2,  _( "90-deg" ));
	table.setHeader( 3,  _( "45-deg" ));
	if ( gradkreis > GRADKREIS_45 ) table.setHeader( 4,  _( "22.5-deg" ));
	if ( gradkreis > GRADKREIS_225 ) table.setHeader( 5,  _( "11.25-deg" ));
	for( i = 0; i < plist.size(); i++ )
	{
		len = p[i];
		table.setEntry( 0, i+1, lang->getOListObjectName( plist[i], TLARGE ) );
		table.setEntry( 1, i+1, formatter->getPosFormatted( len ) );

		len = a_red( len, 90 );
		table.setEntry( 2, i+1, formatter->getDegreesFormated( len ) );

		len = a_red( len, 45 );
		table.setEntry( 3, i+1, formatter->getDegreesFormated( len ));

		if ( gradkreis > GRADKREIS_45 )
		{
			len = a_red( len, 22.5 );
			table.setEntry( 4, i+1, formatter->getDegreesFormated( len ));
		}

		if ( gradkreis > GRADKREIS_225 )
		{
			len = a_red( len, 11.25 );
			table.setEntry( 5, i+1, formatter->getDegreesFormated( len ));
		}
	}
	writer->writeTable( table ); 
}

/*****************************************************
**
**   MidpointExpert   ---   writeConjunctions 
**
******************************************************/
void MidpointExpert::writeConjunctions( Writer *writer, bool show_datum )
{
	Formatter *formatter = Formatter::get();
	Lang *lang = Lang::get();
	list<ConjunctionEvent>::iterator iter;
	vector<wxString> v;
	wxString s;

  writer->writeHeader2( _( "Conjunctions" ));

	if ( conjunction_events.size() <= 8 )
	{
		Table table( 3, conjunction_events.size()+1 );

		wxString header = ( show_datum ? _( "Date" ) : _( "Orbis" ));
		table.setHeader( 0, _( "No." ));
		table.setHeader( 1, _( "Event" ));
		table.setHeader( 2, header );
		table.col_alignment[0] = TABLE_ALIGN_RIGHT;
		table.col_alignment[1] = TABLE_ALIGN_CENTER;
		table.col_alignment[2] = TABLE_ALIGN_RIGHT;
		int line = 1;
		for( iter = conjunction_events.begin(); iter != conjunction_events.end(); iter++ )
		{
			s.Printf( wxT( "%d" ), line );
			table.setHeaderEntry( 0, line, s );
			s.Printf( wxT( "%s %s %s" ), lang->getOListObjectName( (*iter).p1, TSHORT ).c_str(), getEventSymbol(writer, (*iter).type),
				lang->getOListObjectName( (*iter).p2, TSHORT ).c_str() );
			table.setEntry( 1, line, s );
			if ( show_datum )
			{
				if ( (*iter).jd == 0 ) s = wxT( "---" );
				else s = formatter->getDateStringFromJD( (*iter).jd );
			}
			else
			{
				s.Printf( wxT( "%s%s" ), ( (*iter).orbis < 0 ? wxT( "-" ) : wxT( " " )),
					formatter->getDegreesFormated( fabs( (*iter).orbis )).c_str());
			}
			table.setEntry( 2, line, s );
			line++;
		}
		writer->writeTable( table );
	}
	else
	{
		int nb_lines = conjunction_events.size() / 4;
		if ( conjunction_events.size() % 4 ) nb_lines++;
		Table table( 15, nb_lines+1 );

		wxString header = ( show_datum ? _( "Date" ) : _( "Orbis" ));
		table.setHeader( 0, _( "No." ));
		table.setHeader( 1, _( "Event" ));
		table.setHeader( 2, header );
		table.setHeader( 3, wxEmptyString );
		table.setHeader( 4, _( "No." ));
		table.setHeader( 5, _( "Event" ));
		table.setHeader( 6, header );
		table.setHeader( 7, wxEmptyString );
		table.setHeader( 8, _( "No." ));
		table.setHeader( 9, _( "Event" ));
		table.setHeader( 10, _( "Orbis" ));
		table.setHeader( 11, wxEmptyString );
		table.setHeader( 12, _( "No." ));
		table.setHeader( 13, _( "Event" ));
		table.setHeader( 14, header );
		table.col_alignment[0] = TABLE_ALIGN_RIGHT;
		table.col_alignment[1] = TABLE_ALIGN_CENTER;
		table.col_alignment[2] = TABLE_ALIGN_RIGHT;
		table.col_alignment[4] = TABLE_ALIGN_RIGHT;
		table.col_alignment[5] = TABLE_ALIGN_CENTER;
		table.col_alignment[6] = TABLE_ALIGN_RIGHT;
		table.col_alignment[8] = TABLE_ALIGN_RIGHT;
		table.col_alignment[9] = TABLE_ALIGN_CENTER;
		table.col_alignment[10] = TABLE_ALIGN_RIGHT;
		table.col_alignment[12] = TABLE_ALIGN_RIGHT;
		table.col_alignment[13] = TABLE_ALIGN_CENTER;
		table.col_alignment[14] = TABLE_ALIGN_RIGHT;
		int line = 1;
		int number = 1;
		int col = 0;
		for( int j = 0; j < nb_lines; j++ )
		{
			table.setEntry( 3, j+1, wxEmptyString );
			table.setEntry( 7, j+1, wxEmptyString );
			table.setEntry( 11, j+1, wxEmptyString );
		}
		for( iter = conjunction_events.begin(); iter != conjunction_events.end(); iter++ )
		{
			s.Printf( wxT( "%d" ), number );
			//table.setEntry( 4*col, line, s );
			table.setHeaderEntry( 4*col, line, s );
			s.Printf( wxT( "%s %s %s" ), lang->getOListObjectName( (*iter).p1, TSHORT ).c_str(), getEventSymbol(writer, (*iter).type),
				lang->getOListObjectName( (*iter).p2, TSHORT ).c_str() );
			table.setEntry( 4*col+1, line, s );
			if ( show_datum )
			{
				if ( (*iter).jd == 0 ) s = wxT( "---" );
				else s = formatter->getDateStringFromJD( (*iter).jd );
			}
			else
			{
				s.Printf( wxT( "%s%s" ), ( (*iter).orbis < 0 ? wxT( "-" ) : wxT( " " )), 
					formatter->getDegreesFormated( fabs( (*iter).orbis )).c_str());
			}
			table.setEntry( 4 * col +2, line, s );
			number++;
			line++;
			if ( line > nb_lines )
			{
				line = 1;
				col++;
			}
		}
		writer->writeTable( table );
	}
}

/*****************************************************
**
**   MidpointExpert   ---   calculateEvents 
**
******************************************************/
void MidpointExpert::calculateEvents( const int &which1, const int &which2, const double &orbis,
	const int gradkreis, const int order, const int filter  )
{
	unsigned int i, j, k;
	int type;
	double dist, midpoint;

	double *p1 = which1 == 1 ? pos1 : pos2;
	double *p2 = which2 == 1 ? pos1 : pos2;
	midpoint_events.clear();
	conjunction_events.clear();
	midpoint_values.clear();
	for( i = 0; i < plist.size(); i++ )
	{
		for( j = i + 1; j < plist.size(); j++ )
		{
			midpoint = red_deg( ( p2[i] + p2[j] ) * .5 );
			midpoint_values.push_back( MidpointValue( plist[i], i, plist[j], j, midpoint ));
			for( k = 0; k < plist.size(); k++ )
			{
				if ( k == i || k == j ) continue;
				type = berechneNGradTreffer( midpoint, p1[k], (int)orbis, dist, gradkreis );
				if ( type && ( filter == -1 || (int)k == filter || (int)j == filter || (int)i == filter ))
				{
					midpoint_events.push_back( MidpointEvent( plist[k], k, plist[i], i, plist[j], j, dist, type ));
				}
			} // for k
		} // for j
	}  // for i
	midpoint_events.sort( MidPointSort( order ) );

	for( i = 0; i < plist.size(); i++ )
	{
		for( j = 0; j < plist.size(); j++ )
		{
			if ( which1 == which2 && j <= i ) continue;

			type = berechneNGradTreffer( p2[j], p1[i], (int)orbis, dist, gradkreis );
			if ( type && ( filter == -1 || (int)j == filter || (int)i == filter ))
			{
				conjunction_events.push_back( ConjunctionEvent( plist[i], i, plist[j], j, dist, type ));
			}
		}
	}
	conjunction_events.sort( KonjSort( order ));
}

struct dentry {
	vector<wxString> a;
};

/*****************************************************
**
**   MidpointExpert   ---   writeMidpointDiagram 
**
******************************************************/
void MidpointExpert::writeMidpointDiagram( Writer *writer, const int &which1, const int &which2, const double &orbis,
	const int gradkreis, const int filter )
{
	int i, j, k, type;
	double dist, midpoint;
	vector<MidpointDiagram> mp;
	Lang *lang = Lang::get();

	int imax = ( hsSystem ? (int)plist.size()-1 : 12 );
	double *p1 = which1 == 1 ? pos1 : pos2;
	double *p2 = which2 == 1 ? pos1 : pos2;

	writer->writeHeader2( _( "Diagram" ));
	for( i = 0; i < (int)plist.size(); i++ )
	{
		mp.push_back( MidpointDiagram());
		mp[i].master = lang->getOListObjectName( plist[i], TMEDIUM );

		for( j = 0; j <= imax; j++ )
		{
			if ( j == i ) continue;
			for( k = j + 1; k <= imax; k++ )
			{
				if ( k == i ) continue;
				midpoint = red_deg( ( p2[j] + p2[k] ) * .5 );
				type = berechneNGradTreffer( p1[i], midpoint, (int)orbis, dist, gradkreis );
				if ( ! type || fabs( dist * 60 ) > orbis ) continue;
				if ( filter != -1 && k != filter && j != filter && i != filter ) continue;
				mp[i].entries.push_back( MidpointDiagramEntry( lang->getOListObjectName( plist[j], TSHORT ),
					lang->getOListObjectName( plist[k], TSHORT )));
			}
		}
	}
	writer->writeMidpointDiagram( mp );
}


/*****************************************************
**
**   MidpointExpert   ---   writeMidpointEventsText 
**
******************************************************/
void MidpointExpert::writeMidpointEventsText( Writer *writer, const bool show_datum )
{
	wxChar buf[4096], filename[1024];
	list<MidpointEvent>::iterator iter;
	list<ConjunctionEvent>::iterator citer;
	wxString s;
	if ( hsSystem )
		wxSprintf( filename, wxT( "%sregel1.dat" ), Session::get()->getBinDir().c_str() );
	else 
		wxSprintf( filename, wxT( "%skdg1.dat" ), Session::get()->getBinDir().c_str() );
	if ( wxAccess( filename, F_OK )) return;
	if ( wxAccess( filename, R_OK )) return;

	writer->writeHeader2( _( "Text Output" ));
	for( citer = conjunction_events.begin(); citer != conjunction_events.end(); citer++ )
	{
		s.Clear();
		(*citer).write( writer, s, show_datum );
		writer->writeHeader3( s );
		getTextOutput( buf, filename, -1, (*citer).p1, (*citer).p2 );
		writer->writeLine( buf );
	}
	for( iter = midpoint_events.begin(); iter != midpoint_events.end(); iter++ )
	{
		s.Clear();
		(*iter).write( writer, s, show_datum );
		writer->writeHeader3( s );
		getTextOutput( buf, filename, (*iter).p, (*iter).h1, (*iter).h2 );
		writer->writeLine( buf );
	}
}

/*****************************************************
**
**   MidpointExpert   ---   writeMidpointEvents 
**
******************************************************/
void MidpointExpert::writeMidpointEvents( Writer *writer, bool show_datum )
{
	list<MidpointEvent>::const_iterator iter;
	vector<wxString> v;
	wxString s;
	Formatter *formatter = Formatter::get();
	Lang *lang = Lang::get();


  writer->writeHeader2( _( "Midpoints" ));
	int nb_lines = midpoint_events.size() / 4;
	if ( midpoint_events.size() % 4 ) nb_lines++;
	Table table( 15, nb_lines+1 );

	wxString header = ( show_datum ? _( "Date" ) : _( "Orbis" ));
	table.setHeader( 0, _( "No." ));
	table.setHeader( 1, _( "Event" ));
	table.setHeader( 2, header );
	table.setHeader( 3, wxEmptyString );
	table.setHeader( 4, _( "No." ));
	table.setHeader( 5, _( "Event" ));
	table.setHeader( 6, header );
	table.setHeader( 7, wxEmptyString );
	table.setHeader( 8, _( "No." ));
	table.setHeader( 9, _( "Event" ));
	table.setHeader( 10, header );
	table.setHeader( 11, wxEmptyString );
	table.setHeader( 12, _( "No." ));
	table.setHeader( 13, _( "Event" ));
	table.setHeader( 14, _( "Orbis" ));
	table.col_alignment[0] = TABLE_ALIGN_RIGHT;
	table.col_alignment[1] = TABLE_ALIGN_CENTER;
	table.col_alignment[2] = TABLE_ALIGN_RIGHT;
	table.col_alignment[4] = TABLE_ALIGN_RIGHT;
	table.col_alignment[5] = TABLE_ALIGN_CENTER;
	table.col_alignment[6] = TABLE_ALIGN_RIGHT;
	table.col_alignment[8] = TABLE_ALIGN_RIGHT;
	table.col_alignment[9] = TABLE_ALIGN_CENTER;
	table.col_alignment[10] = TABLE_ALIGN_RIGHT;
	table.col_alignment[12] = TABLE_ALIGN_RIGHT;
	table.col_alignment[13] = TABLE_ALIGN_CENTER;
	table.col_alignment[14] = TABLE_ALIGN_RIGHT;
	int line = 1;
	int number = 1;
	int col = 0;
	for( int j = 0; j < nb_lines; j++ )
	{
		table.setEntry( 3, j+1, wxEmptyString );
		table.setEntry( 7, j+1, wxEmptyString );
		table.setEntry( 11, j+1, wxEmptyString );
	}
	for( iter = midpoint_events.begin(); iter != midpoint_events.end(); iter++ )
	{
		s.Printf( wxT( "%d" ), number );
		//table.setEntry( 4*col, line, s );
		table.setHeaderEntry( 4*col, line, s );
		s.Printf( wxT( "%s %s %s/%s" ), lang->getOListObjectName( (*iter).p, TSHORT ).c_str(),
			getEventSymbol(writer, (*iter).type),
			lang->getOListObjectName( (*iter).h1, TSHORT ).c_str(),
			lang->getOListObjectName( (*iter).h2, TSHORT ).c_str() );
		table.setEntry( 4*col+1, line, s );
		if ( show_datum )
		{
			if ( (*iter).jd == 0 ) s = wxT( "---" );
			else s = formatter->getDateStringFromJD( (*iter).jd );
		}
		else
		{
			s.Printf( wxT( "%s%s" ), ( (*iter).orbis < 0 ? wxT( "-" ) : wxT( " " )), 
				formatter->getDegreesFormated( fabs( (*iter).orbis )).c_str());
		}
		table.setEntry( 4 * col +2, line, s );
		number++;
		line++;
		if ( line > nb_lines )
		{
			line = 1;
			col++;
		}
	}
	writer->writeTable( table );
}

/*****************************************************
**
**   MidpointExpert   ---   getTextOutput 
**
******************************************************/
void MidpointExpert::getTextOutput( wxChar *buf, wxChar *filename, const int &p, const int &h1, const int &h2 )
{
	char tmp[4096], tmp2[4096], code[32];
	int count = 0;
	char *c;

	wxStrcpy( buf, wxT( "not found\n" ));
	if ( p == h1 || h1 == h2 || h2 == p )
		{ wxStrcpy( buf, wxT( "-\n" )); return; }

	if ( hsSystem )
	{
		getPlanetaryIndexW( code, p, h1, h2 );
	}
	else
	{
		if ( p == -1 ) { wxStrcpy( buf, wxT( "-" )); return; }
		sprintf( code, "%d" , mapPlanetsToNumberE( p, h1, h2 ));
	}
	FILE *fh = fopen( wxConvertWX2MB( filename ),  "r" );
	if ( ! fh ) { wxStrcpy( buf, wxT( "Text not found (1)" )); return; }

  while( fgets( tmp, 4096, fh ) && count++ < 6000 )
	{
		c = strchr( tmp, '\t' );
		if( ! c ) { wxStrcpy( buf, wxT( "Text not found (2)" )); break; }
		*c = 0;
		if ( ! strcmp( code, tmp ))
		{
			strcpy( tmp2, c+1 );
			wxStrcpy( buf, wxConvertMB2WX( tmp2 ));
			///wxStrcpy( buf, wxConvertMB2WX( c+1 ) );
			break;
		}
	}
	fclose( fh );
	assert( count < 6000 );
}

/*****************************************************
**
**   MidpointExpert   ---   writeChainAnalysis
**
******************************************************/
void MidpointExpert::writeChainAnalysis( Writer *writer, Horoscope *h, const double &orbis,
	const int gradkreis, const int order, const int viewmodus )
{
	unsigned int i;
	int type, mytreffer;
	wxString s;
	double dist;
	Formatter *formatter = Formatter::get();
	Lang *lang = Lang::get();
	list<MidpointValue>::iterator iter, iter2;

	updatePos( 1, h->getDataSet() );
	writer->writeHeader1( _( "Chains" ));
	s << _( "Circle" ) << wxT( ": " )  << gradkreis_name[gradkreis] << wxT( " " ) << _( "degrees" );
	writer->writeLine( s );
	s.Clear();
	s << _( "Orbis" ) << wxT( ":" ) << orbis << wxT( " " ) << _( "min" );
	writer->writeParagraph( s );
	s.Clear();
	calculateEvents( 1, 1, orbis, gradkreis, order );

	for( i = 0; i < plist.size(); i++ )
	{
		s.Clear();
		mytreffer = 0;
		for( iter = midpoint_values.begin(); iter != midpoint_values.end(); iter++ )
		{
			type = berechneNGradTreffer( pos1[i], (*iter).wert, (int)orbis, dist, gradkreis );
			if ( ! type ) continue;
			if ( mytreffer == 0 )
			{
				if ( viewmodus == 0 ) { s << lang->getOListObjectName( plist[i], TSHORT ); }
				else { writer->writeHeader2 ( lang->getOListObjectName( plist[i], TLARGE ) ); }
			}
			mytreffer++;
			if ( viewmodus == 0 )
			{
				s << wxT( " " ) << getEventSymbol(writer, type) << wxT( " " ) << lang->getOListObjectName( (*iter).p1, TSHORT )
					<< wxT( "/" ) << lang->getOListObjectName( (*iter).p2, TSHORT );
			}
			else
			{
				s.Printf( wxT( " %c %s/%s %c%s" ), getEventSymbol(writer, type), lang->getOListObjectName( (*iter).p1, TSHORT ).c_str(),
					lang->getOListObjectName( (*iter).p2, TSHORT ).c_str(), ( dist < 0 ? '-' : ' ' ), 
					formatter->getDegreesFormated( fabs(dist)).c_str());
				writer->writeLine( s );
			}
		}
		if ( viewmodus == 0 )
		{ if ( mytreffer ) writer->writeParagraph( s ); }
	}
	for( iter2 = midpoint_values.begin(); iter2 != midpoint_values.end(); iter2++ )
	{
		s.Clear();
		mytreffer = 0;
		for( iter = midpoint_values.begin(); iter != midpoint_values.end(); iter++ )
		{
			if ( iter == iter2 ) continue;
			type = berechneNGradTreffer( (*iter2).wert, (*iter).wert, (int)orbis, dist, gradkreis );
			if ( ! type ) continue;
			if ( mytreffer == 0 )
			{
				if ( viewmodus == 0 )
				{
					s << lang->getOListObjectName( (*iter2).p1, TSHORT ) << wxT( "/" )
						<< lang->getOListObjectName( (*iter2).p2, TSHORT );
				}
				else
				{
					s.Printf( wxT( "%s/%s" ), lang->getOListObjectName( (*iter2).p1, TSHORT ).c_str(),
						lang->getOListObjectName( (*iter2).p2, TSHORT ).c_str());
					writer->writeHeader2( s );
				}
			}
			mytreffer++;
			if ( viewmodus == 0 )
			{
				s << wxT( " " ) << getEventSymbol(writer, type) << wxT( " " ) << lang->getOListObjectName( (*iter).p1, TSHORT )
					<< wxT( "/" ) << lang->getOListObjectName( (*iter).p2, TSHORT );
			}
			else
			{
				s.Printf( wxT( "%c %s/%s %c%s" ), getEventSymbol(writer, type), lang->getOListObjectName( (*iter).p1, TSHORT ).c_str(),
					lang->getOListObjectName( (*iter).p2, TSHORT ).c_str(),
					( dist < 0 ? '-' : ' ' ), 
					formatter->getDegreesFormated( fabs(dist) ).c_str());
				writer->writeLine( s );
			}
		}
		for( i = 0; i < plist.size(); i++ )
		{
			type = berechneNGradTreffer( (*iter2).wert, pos1[i], (int)orbis, dist, gradkreis );
			if ( ! type ) continue;
			if ( mytreffer == 0 )
			{
				if ( viewmodus == 0 )
				{
					s << lang->getOListObjectName( (*iter2).p1, TSHORT ) << wxT( "/" )
						<< lang->getOListObjectName( (*iter2).p2, TSHORT );
				}
				else
				{
					s.Printf( wxT( "%s/%s" ), lang->getOListObjectName( (*iter2).p1, TSHORT ).c_str(),
						lang->getOListObjectName( (*iter2).p2, TSHORT ).c_str());
					writer->writeHeader2( s );
				}
			}
			mytreffer++;
			if ( viewmodus == 0 )
			{
				s << wxT( " " ) << getEventSymbol(writer, type) << wxT( " " ) << lang->getOListObjectName( i, TSHORT );
			}
			else
			{
				s.Printf( wxT( "%c %s %c%s" ), getEventSymbol(writer, type), lang->getOListObjectName( i, TSHORT ).c_str(),
					( dist < 0 ? '-' : ' ' ), 
					formatter->getDegreesFormated( fabs(dist) ).c_str());
				writer->writeLine( s );
			}
		}
		if ( viewmodus == 0 )
		{ if ( mytreffer ) writer->writeParagraph( s ); }
	}
}

/*****************************************************
**
**   MidpointExpert   ---   calcSolarArcEvent 
**
******************************************************/
double MidpointExpert::calcSolarArcEvent( DataSet *basedate, const double &aprox_jd, const double &red, const int &planet,
	const double &targetpos, const int &gradkreis )
{
	int type;
	const int orbis = 90;
	int count = 0;
	double sunlen, sunspeed, dummy, bogen, dist = 100, transitlen, radixlen, mclen;
	Calculator *calculator = Session::get()->getCalculator();

	double ayanamsa = calculator->calcAyanamsa( basedate->getJD(), config->wAyanamsa );
	DataSet d( *basedate );

	radixlen = pos1[planet];
	mclen = pos1[mcpos];
	d.setDate( basedate->getJD() + ( aprox_jd - basedate->getJD() ) / red );

	while( fabs( dist ) > .000001 )
	{
		//calculator->calcPosSpeedAya( jd, OSUN, sunlen, dummy, sunspeed, false );
		calculator->calcPositionSpeed( &d, OSUN, sunlen, dummy, sunspeed, true, false );
		bogen = red_deg( sunlen - pos1[sunpos] );

		if ( planet != (int)aspos )
		{
			transitlen = red_deg( radixlen + bogen );
			type = berechneNGradTreffer( targetpos, transitlen, orbis, dist, gradkreis );
			d.setDate( d.getJD() - dist );
		}
		else
		{
			transitlen = red_deg( calcAscendantFromMC( d.getJD(), red_deg( mclen + ayanamsa + bogen ),
				basedate->getLocation()->getLatitude() ) - ayanamsa );
			type = berechneNGradTreffer( targetpos, transitlen, orbis, dist, gradkreis );
			d.setDate( d.getJD() - dist/2 );
		}

		count++;
		assert( count < 160 ); // normally about 5 but can be more for ascendant
	}
	return basedate->getJD() + red * ( d.getJD() - basedate->getJD() );
}

/*****************************************************
**
**   MidpointExpert   ---   writeYearlyPreviewAnalysis
**
******************************************************/
void MidpointExpert::writeYearlyPreviewAnalysis( Writer *writer, DataSet *basedate, const double &red,
	const int &gradkreis, const int &year )
{
	unsigned int i;
	int theyear, dummy1, dummy2;
	double bogen, len, dummy;
	const double orbis = 45;
	wxString s;
	list<ConjunctionEvent>::iterator c_iter, cd_iter;
	list<MidpointEvent>::iterator m_iter, md_iter;

	Formatter *formatter = Formatter::get();
	Calculator *calculator = Session::get()->getCalculator();

	Location *loc = basedate->getLocation();
	conjunction_events.clear();
	writer->writeHeader1( _( "Yearly Preview" ));
	s << _( "Year" ) << wxT( ": " ) << year;
	writer->writeLine( s );
	s.Clear();
	s <<  _( "Year Length" ) << wxT( ": " ) << red;
	writer->writeParagraph( s );
	s.Clear();
	updatePos( 1, basedate );

	DataSet transitdate( *basedate );
	transitdate.setDate( 1, 7, year, 0 ); // jd of 1st july

	//jd = basedate->getJD() + ( transitdate.getJD() - basedate->getJD() ) / red;
	// transitdate.setDate( basedate->getJD() + ( transitdate.getJD() - basedate->getJD() ) / red );
	DataSet date2( *basedate );
	date2.setDate( basedate->getJD() + ( transitdate.getJD() - basedate->getJD() ) / red );

	//bogen = red_deg( calculator->calcPosAya( jd, OSUN, false ) - pos1[sunpos] );
	calculator->calcPosition( &date2, OSUN, len , dummy, false );
	bogen = red_deg( len - pos1[sunpos] );

	for( i = 0; i < plist.size(); i++ )
	{
		if ( i == aspos ) continue;
		pos2[i] = red_deg( pos1[i] + bogen );
	}
	pos2[aspos] = calcAscendantFromMC( transitdate.getJD(), pos2[mcpos], loc->getLatitude() );

	calculateEvents( 2, 1, orbis, gradkreis, MIDPOINT_SORT_ORBIS_REV );
	for( c_iter = conjunction_events.begin(); c_iter != conjunction_events.end(); c_iter++ )
	{
		(*c_iter).jd = calcSolarArcEvent( basedate, transitdate.getJD(), red, (*c_iter).i1, pos1[(*c_iter).i2], gradkreis );
	}
	c_iter = conjunction_events.begin();
	while( c_iter != conjunction_events.end())  // delete entries with wrong year
	{
		formatter->getDateIntsFromJD( (*c_iter).jd, dummy1, dummy2, theyear );
		if ( theyear != year )
		{
			cd_iter = c_iter; // -> don't delete the one you want to increment.
			c_iter++;
			conjunction_events.erase( cd_iter );
		}
		else c_iter++;
	}
	conjunction_events.sort( KonjSort( MIDPOINT_SORT_JD ));
	writeConjunctions( writer, true );

	for( m_iter = midpoint_events.begin(); m_iter != midpoint_events.end(); m_iter++ )
	{
		(*m_iter).jd = calcSolarArcEvent( basedate, transitdate.getJD(), red, (*m_iter).i0,
			a_red( .5 * ( pos1[(*m_iter).i1] + pos1[(*m_iter).i2]), getKreisgrad( gradkreis )), gradkreis );
	}
	m_iter = midpoint_events.begin();
	while( m_iter != midpoint_events.end())
	{
		formatter->getDateIntsFromJD( (*m_iter).jd, dummy1, dummy2, theyear );
		if ( theyear != year )
		{
			md_iter = m_iter; // -> don't delete the one you want to increment.
			m_iter++;
			midpoint_events.erase( md_iter );
		}
		else m_iter++;
	}
	midpoint_events.sort( MidPointSort( MIDPOINT_SORT_JD ) );
	writeMidpointEvents( writer, true );
	writeMidpointEventsText( writer, true );
}

/*****************************************************
**
**   MidpointExpert   ---   writeDirectionAnalysis
**
******************************************************/
void MidpointExpert::writeDirectionAnalysis( Writer *writer, DataSet *basedate, DataSet *transitdate,
	const double &red, const double &orbis, const int gradkreis, const int order, const int filter )
{
	double diffjd, jd;
	Formatter *formatter = Formatter::get();
	wxString s;

	updatePos( 1, basedate );
	writer->writeHeader1( _( "Directions" ));
	s << _( "Year Length" ) << wxT( ": " ) << red;
	writer->writeLine( s );
	s.Clear();
	s << _( "Event Date" ) << wxT( ": " ) <<  formatter->getFullDateStringFromJD( transitdate->getJD() ) << Endl;
	writer->writeLine( s );
	s.Clear();
	s << _( "Orbis" ) << wxT( ": " ) << orbis << wxT( " " ) << _( "min" ) << Endl;
	writer->writeLine( s );
	s.Clear();

	diffjd = transitdate->getJD() - basedate->getJD();
	diffjd /= red;

	transitdate->setDate( basedate->getJD() + diffjd );
	updatePos( 2, transitdate );

	s << _( "Calculated Date: " ) << formatter->getFullDateStringFromJD( jd ) << Endl;
	writer->writeLine( s );
	s.Clear();

	writer->writeHeader2( _( "Positions" ));
	writePlanetaryPositions( writer, 2, gradkreis );
	calculateEvents( 2, 1, orbis, gradkreis, order );
	writeConjunctions( writer );
	writeMidpointEvents( writer ),
	writeMidpointDiagram( writer, 2, 1, orbis, gradkreis );
	writeMidpointEventsText( writer );
}

/*****************************************************
**
**   MidpointExpert   ---   writeConstantArcAnalysis
**
******************************************************/
void MidpointExpert::writeConstantArcAnalysis( Writer *writer, DataSet *basedate, DataSet *transitdate,
	const double &red, const double &orbis, const int gradkreis, const int order )
{
	wxString s;
	writer->writeHeader1( _( "Constant Arc" ));
	s <<  _( "Year Length" ) << wxT( ": " ) << red << Endl;
	writer->writeLine( s );
	s.Clear();
	double bogen = ( transitdate->getJD() - basedate->getJD()  ) / red;
	writeArcEventAnalysis( writer, bogen, basedate, transitdate, red, orbis, gradkreis, order );
}

/*****************************************************
**
**   MidpointExpert   ---   writeSolarArcAnalysis
**
******************************************************/
void MidpointExpert::writeSolarArcAnalysis( Writer *writer, DataSet *basedate, DataSet *transitdate,
	const double &red, const double &orbis, const int gradkreis, const int order, const int filter )
{
	writePlanetaryArcAnalysis( writer, OSUN, basedate, transitdate, red, orbis, gradkreis, order, filter );
}

/*****************************************************
**
**   MidpointExpert   ---   calcAscendantFromMC 
**
******************************************************/
double MidpointExpert::calcAscendantFromMC( const double &jd, const double &mc, const double &breite )
{
	double eps, armc;
	Calculator *calculator = Session::get()->getCalculator();

	// x[0] enthaelt eps
	calculator->calcEps( eps, jd );

	// armc des vorgeschobenen MC berechnen
	armc = atan( tan( mc * DEG2RAD ) * cos( eps * DEG2RAD )) * RAD2DEG;
	if ( mc > 90 && mc < 270 ) armc += 180;
	armc = red_deg( armc );

	return calculator->calcAscendantByArmc( armc, eps, breite );
}

/*****************************************************
**
**   MidpointExpert   ---   writeArcEventAnalysis
**
******************************************************/
double MidpointExpert::writeArcEventAnalysis( Writer *writer, const double &bogen, DataSet *basedate,
	DataSet *transitdate, const double &red, const double &orbis, const int gradkreis,
	const int order, const int filter )
{
	double jd;
	unsigned int i;
	Formatter *formatter = Formatter::get();
	wxString s;

	updatePos( 1, basedate );

	s.Printf( wxT( "%s: %s" ),  _( "Event date" ), formatter->getFullDateStringFromJD( transitdate->getJD() ).c_str() );
	writer->writeLine( s );
	s.Printf( wxT( "%s: %s" ),  _( "Location" ), transitdate->getLocation()->getLocName().c_str() );
	writer->writeLine( s );
	s.Printf( wxT( "%s: %1.1f %s" ),  _( "Orbis" ), orbis, wxT( "min" ));
	writer->writeParagraph( s );
	s.Clear();
	s << _( "Circle" ) << wxT( ": " ) << gradkreis_name[gradkreis] << wxT( " " ) << _(  "degrees" );
	writer->writeLine( s );
	s.Clear();
	s << _( "Year Length" ) << wxT( ": " ) << red;
	writer->writeParagraph( s );
	s.Clear();

	jd = basedate->getJD() + ( transitdate->getJD() - basedate->getJD() ) / red;

	s.Printf( wxT( "%s: %s" ),  _( "Calculated Date" ), formatter->getFullDateStringFromJD( jd ).c_str() );
	writer->writeLine( s );
	s.Printf( wxT( "%s: %s" ),  _( "Arc" ), formatter->getDegreesFormated( bogen ).c_str() );
	writer->writeParagraph( s );

	for( i = 0; i < plist.size(); i++ )
	{
		if ( plist[i] == OASCENDANT ) continue;
		pos2[i] = red_deg( pos1[i] + bogen );
	}
	//pos2[aspos] = calcAscendantFromMC( jd, pos2[mcpos], transitdate->getLocation()->getLatitude() );
	pos2[aspos] = red_deg( pos1[aspos] + bogen );

	writer->writeHeader2( _( "Calculated Positions" ) );
	writePlanetaryPositions( writer, 2, gradkreis );
	calculateEvents( 2, 1, orbis, gradkreis, order, filter );
	writeConjunctions( writer );
	writeMidpointEvents( writer );
	writeMidpointDiagram( writer, 2, 1, orbis, gradkreis, filter );
	writeMidpointEventsText( writer );
	return jd;
}

/*****************************************************
**
**   MidpointExpert   ---   writePlanetaryArcAnalysis
**
******************************************************/
double MidpointExpert::writePlanetaryArcAnalysis( Writer *writer, const int &planet, DataSet *basedate,
	DataSet *transitdate, const double &red, const double &orbis, const int gradkreis, const int order, const int filter )
{
	double ppos1, ppos2, dummy;
	Calculator *calculator = Session::get()->getCalculator();

	assert( planet == OSUN || planet == OMOON );
	if ( planet == OMOON ) writer->writeHeader1( _( "Lunar Arc Analysis" ));
	else writer->writeHeader1( _( "Solar Arc Analysis" ));

	calculator->calcPosition( basedate, planet, ppos1, dummy, true, false );
	DataSet d( *transitdate );
	d.setDate( basedate->getJD() + ( transitdate->getJD() - basedate->getJD() ) / red );
	calculator->calcPosition( &d, planet, ppos2, dummy, true, false );
	return writeArcEventAnalysis( writer, red_deg( ppos2 - ppos1 ), basedate, transitdate, red,
		orbis, gradkreis, order, filter );
}

/*****************************************************
**
**   MidpointExpert   ---   writeTransitAnalysis
**
******************************************************/
void MidpointExpert::writeTransitAnalysis( Writer *writer, DataSet *basedate, DataSet *transitdate,
	const double &orbis, const int gradkreis, const int order )
{
	wxString s;
	updatePos( 1, basedate );
	updatePos( 2, transitdate );

	writer->writeHeader1( _( "Transits" ));
	Formatter *formatter = Formatter::get();

	s.Printf( wxT( "%s: %s" ), _( "Date" ), formatter->getFullDateStringFromJD( transitdate->getJD() ).c_str() );
	writer->writeLine( s );
	s.Printf( wxT( "%s: %1.1f %s" ), _( "Orbis" ), orbis, _( "min" ));
	writer->writeLine( s );

	writer->writeHeader2( _( "Transit Positions" ));
	writePlanetaryPositions( writer, 2, gradkreis );

	calculateEvents( 2, 1, orbis, gradkreis, order );
	writeConjunctions( writer );
	writeMidpointEvents( writer );
	writeMidpointDiagram( writer, 2, 1, orbis, gradkreis );
	writeMidpointEventsText( writer );
}

/*****************************************************
**
**   MidpointExpert   ---   writePartnerAnalysis
**
******************************************************/
void MidpointExpert::writePartnerAnalysis( Writer *writer, Horoscope *partner1, Horoscope *partner2,
	const double &orbis, const int gradkreis, const int order, const int filter )
{
	wxString s;
	double tjd;
	Location *loc;
	Formatter *formatter = Formatter::get();
	
	updatePos( 1, partner1->getDataSet() );
	updatePos( 2, partner2->getDataSet() );

	writer->writeHeader1( _( "Partner Horoscope" ));
	s << partner1->getHName() << wxT( " + " ) << partner2->getHName();
	writer->writeLine( s );
	s.Clear();
	s << _( "Orbis" ) << wxT( ": " ) << orbis << wxT( " " ) << _( "min" ) << Endl;
	writer->writeLine( s );
	s.Clear();

	tjd = partner1->getJD();
	loc = partner1->getLocation();
	s << _( "Partner 1" ) << wxT( ": " ) << partner1->getHName();
	writer->writeHeader1( s );
	s.Clear();
	s.Printf( wxT( "%s: %s" ), _( "Date" ), 
		formatter->getFullDateStringFromJD( tjd + ( loc->getTimeZone()+loc->getDST() ) / 24   ).c_str());
	writer->writeLine( s );

	calculateEvents( 2, 1, orbis, gradkreis, order, filter );
	writeConjunctions( writer );
	writeMidpointEvents( writer );
	writeMidpointDiagram( writer, 2, 1, orbis, gradkreis, filter );
	writeMidpointEventsText( writer );

	tjd = partner2->getJD();
	loc = partner2->getLocation();
	s.Clear();
	s << _( "Partner 2" ) << wxT( ": " ) << partner2->getHName();
	writer->writeHeader1( s );
	s.Clear();
	s.Printf( wxT( "%s: %s" ), _( "Date" ), 
		formatter->getFullDateStringFromJD( tjd + ( loc->getTimeZone()+loc->getDST() ) / 24   ).c_str());
	writer->writeLine( s );

	calculateEvents( 1, 2, orbis, gradkreis, order, filter );
	writeConjunctions( writer );
	writeMidpointEvents( writer );
	writeMidpointDiagram( writer, 2, 1, orbis, gradkreis, filter );
	writeMidpointEventsText( writer );
}

