/*

	This is the source code of

   	M A I T R E Y A
    ===============

	Open source platform for Vedic and western astrology

  File           SimpleHoroscope.cpp
  Release        4.1
  Author         Martin Pettau
  Copyright (C)  2003-2006 by the author

  Released under the Artistic License as published by the
  Free Software Foundation, read the file 'COPYING' for more information.

*/

#ifdef __GNUG__
	#pragma implementation "SimpleHoroscope.h"
#endif

#include "SimpleHoroscope.h"
#include "constants.h"
#include "Calculator.h"
#include "IPlanet.h"
#include "Session.h"
#include "Lang.h"
#include "func.h"
#include "Varga.h"
#include "Conf.h"
#include "Jaimini.h"
#include "VimsottariDasa.h"
#include "Writer.h"

#include <wx/string.h>
#include <assert.h>
#include <stdio.h>
#include <math.h>

extern Config *config;

/*****************************************************
**
**   SimpleHoroscope   ---   Constructor 
**
******************************************************/
SimpleHoroscope::SimpleHoroscope()
{
	int i;

	for ( i = 0; i < MAX_BODIES; i++ )
	{
		object_len[i] = 0;
		object_lat[i] = 0;
		object_speed[i] = 1;
	}
	dataSet = new FileDataSet();
	dataSet->setLocation( config->defaultLocation );
	for( i = 0; i < 12; i++ ) whousecusp[i] = ihousecusp[i] = ihousesandhi[0] = 0;
}

/*****************************************************
**
**   SimpleHoroscope   ---   Destructor 
**
******************************************************/
SimpleHoroscope::~SimpleHoroscope()
{
	delete dataSet;
}

/*****************************************************
**
**   SimpleHoroscope   ---   updatePlanets 
**
******************************************************/
void SimpleHoroscope::updatePlanets()
{
	int i;
	double r[12], houselen;
	Calculator *calculator = Session::get()->getCalculator();

	iayanamsa_type = config->iAyanamsa;
	wayanamsa_type = config->wAyanamsa;
	iayanamsa = calculator->calcAyanamsa( getJD(), iayanamsa_type );
	wayanamsa = calculator->calcAyanamsa( getJD(), wayanamsa_type );
	for( i = OSUN; i <= OPOSEIDON; i++ )
	{
		calculator->calcPositionSpeed( dataSet, i, object_len[i], object_lat[i], object_speed[i] );
	}
	calculator->calcHouses( getJD(), getLongitude(), getLatitude(), config->wHouseSystem, r );
	for( i = 0; i < 12; i++ )
	{
		whousecusp[i] = red_deg( r[i] - wayanamsa );
	}
	if ( config->iHouseSystem != config->wHouseSystem )
	{
		calculator->calcHouses( getJD(), getLongitude(), getLatitude(), config->iHouseSystem, r );
	}
	for( i = 0; i < 12; i++ )
	{
		ihousecusp[i] = red_deg( r[i] - iayanamsa );
		//cout << "Cusp " << i+1 << " " << ihousecusp[i] << Endl;
	}
	for( i = 0; i < 12; i++ )
	{
		houselen = red_deg( ihousecusp[i<11 ? i+1 : 0] - ihousecusp[i] );
		ihousesandhi[i] = red_deg( ihousecusp[i] + .5 * houselen );
	}
	siderealTime = calculator->calcSiderealTime( getJD(), getLongitude() );
}

/*****************************************************
**
**   SimpleHoroscope   ---   getHouse 
**
******************************************************/
double SimpleHoroscope::getHouse( const int& housenb, const bool& vedic, const bool getsandhi )
{
	assert( housenb >= HOUSE1 && housenb <= HOUSE12 );
	if( vedic )
	{
		return( getsandhi ? ihousesandhi[housenb] : ihousecusp[housenb] );
	}
	else
	{
		return whousecusp[housenb];
	}
}

/*****************************************************
**
**   SimpleHoroscope   ---   getHousePos 
**
******************************************************/
int SimpleHoroscope::getHousePos( const int& planet )
{
	assert( planet >= ISUN && planet <= WARIES );
	int i;
	double hstart, hend;
	double len = getObjectLength( planet );
	for( i = HOUSE1; i <= HOUSE12; i++ )
	{
		if ( planet >= WSUN )
		{
			hstart = whousecusp[i];
			hend = whousecusp[red12(i+1)];
		}
		else
		{
			if ( config->iHouseWestern )
			{
				hstart = ihousecusp[i];
				hend = ihousecusp[red12(i+1)];
			}
			else
			{
				hstart = ihousesandhi[red12( i -1)];
				hend = ihousesandhi[red12(i)];
			}
		}
		if (( hstart <= len && len < hend ) || ( hstart > hend && ( len >= hstart || len < hend ))) return red12( i );
	}
	assert( 0 );

return 0;
}

/*****************************************************
**
**   SimpleHoroscope   ---   openFile 
**
******************************************************/
bool SimpleHoroscope::openFile( const wxChar *filename )
{
	return dataSet->openFile( filename );
}


/*****************************************************
**
**   SimpleHoroscope   ---   saveAs 
**
******************************************************/
bool SimpleHoroscope::saveAs( const wxChar *filename )
{
	return dataSet->saveAs( filename );
}


/*****************************************************
**
**   SimpleHoroscope   ---   save 
**
******************************************************/
bool SimpleHoroscope::save()
{
	return dataSet->save();
}

/*****************************************************
**
**   SimpleHoroscope   ---   getObjectLength 
**
******************************************************/
double SimpleHoroscope::getObjectLength( const int &i )
{
	double ret = 0;

	const int inode = config->iLunarNodeMode == LUNAR_NODE_TRUE ? OTRUENODE : OMEANNODE;

	if ( i == ISUN ) ret = object_len[OSUN] - iayanamsa;
	else if ( i == IMOON ) ret = object_len[OMOON] - iayanamsa;
	else if ( i == IMARS ) ret = object_len[OMARS] - iayanamsa;
	else if ( i == IMERCURY ) ret = object_len[OMERCURY] - iayanamsa;
	else if ( i == IJUPITER ) ret = object_len[OJUPITER] - iayanamsa;
	else if ( i == IVENUS ) ret = object_len[OVENUS] - iayanamsa;
	else if ( i == ISATURN ) ret = object_len[OSATURN] - iayanamsa;
	else if ( i == IRAHU ) ret = object_len[inode] - iayanamsa;
	else if ( i == IKETU ) ret = object_len[inode] + 180 - iayanamsa;
	else if ( i == IASCENDANT ) ret = getHouse( HOUSE1, true );

	else if (( i >= WSUN ) && ( i <= WPLUTO )) ret = object_len[i-WSUN] - wayanamsa;
	else if ( i == WLUNARNODE )
	{
		if ( config->wLunarNodeMode == LUNAR_NODE_TRUE ) ret = object_len[OTRUENODE] - wayanamsa;
		else ret = object_len[OMEANNODE] - wayanamsa;
	}
	else if ( i == WASCENDANT ) ret = getHouse( HOUSE1, false );
	else if ( i == WMC ) ret = getHouse( HOUSE10, false );
	else if (( i >= WCUPIDO ) && ( i <= WPOSEIDON )) ret = object_len[i-WMOON] - wayanamsa;
	else if ( i == WARIES ) return 0;
	else assert( 0 );

	return red_deg( ret );
}

/*****************************************************
**
**   SimpleHoroscope   ---   isObjectRetrograde 
**
******************************************************/
bool SimpleHoroscope::isObjectRetrograde( const int &i )
{
	const int inode = config->iLunarNodeMode == LUNAR_NODE_TRUE ? OTRUENODE : OMEANNODE;
	const int wnode = config->wLunarNodeMode == LUNAR_NODE_TRUE ? OTRUENODE : OMEANNODE;

	if ( i == IMARS ) return ( object_speed[OMARS] < 0 );
	else if ( i == IMERCURY ) return ( object_speed[OMERCURY] < 0 );
	else if ( i == IJUPITER ) return ( object_speed[OJUPITER] < 0 );
	else if ( i == IVENUS ) return ( object_speed[OVENUS] < 0 );
	else if ( i == ISATURN ) return ( object_speed[OSATURN] < 0 );
	else if ( i == IRAHU ) return ( object_speed[inode] < 0 );
	else if ( i == IKETU ) return ( object_speed[inode] < 0 );
	else if ( i == WMERCURY ) return ( object_speed[OMERCURY] < 0 );
	else if ( i == WVENUS ) return ( object_speed[OVENUS] < 0 );
	else if ( i == WMARS ) return ( object_speed[OMARS] < 0 );
	else if ( i == WJUPITER ) return ( object_speed[OJUPITER] < 0 );
	else if ( i == WSATURN ) return ( object_speed[OSATURN] < 0 );
	else if ( i == WURANUS ) return ( object_speed[OURANUS] < 0 );
	else if ( i == WNEPTUNE ) return ( object_speed[ONEPTUNE] < 0 );
	else if ( i == WPLUTO ) return ( object_speed[OPLUTO] < 0 );
	else if ( i == WLUNARNODE ) return ( object_speed[wnode] < 0 );
	else if ( i == WCUPIDO ) return ( object_speed[OCUPIDO] < 0 );
	else if ( i == WHADES ) return ( object_speed[OHADES] < 0 );
	else if ( i == WZEUS ) return ( object_speed[OZEUS] < 0 );
	else if ( i == WKRONOS ) return ( object_speed[OKRONOS] < 0 );
	else if ( i == WAPOLLON ) return ( object_speed[OAPOLLON] < 0 );
	else if ( i == WADMETOS ) return ( object_speed[OADMETOS] < 0 );
	else if ( i == WVULKANUS ) return ( object_speed[OVULKANUS] < 0 );
	else if ( i == WPOSEIDON ) return ( object_speed[OPOSEIDON] < 0 );
	return false;
}

/*****************************************************
**
**   SimpleHoroscope   ---   writeAstronomicalData 
**
******************************************************/
void SimpleHoroscope::writeAstronomicalData( Writer *writer )
{
	wxString pname;
	Formatter *f = Formatter::get();
	Lang *l = Lang::get();
	Calculator *calculator = Session::get()->getCalculator();

  writer->writeHeader1( _( "Astronomical Positions" ));
	Table table( 3, 23 );
	table.setHeader( 0,  _( "Planet" ));
	table.setHeader( 1,  _( "Longitude" ));
	table.setHeader( 2,  _( "Latitute" ));
	table.col_line[0] = true;
	table.col_alignment[0] = TABLE_ALIGN_RIGHT;
	table.col_alignment[1] = TABLE_ALIGN_RIGHT;

	int line = OSUN+1;
	for( int i = OSUN; i <= OPOSEIDON; i++ )
	{
		pname = l->getObjectName( l->mapO2WIndex( i ), TLARGE);
		if ( i == OTRUENODE ) pname = _( "True Node" );
		else if ( i == OMEANNODE ) pname = _( "Mean Node" );
		table.setEntry( 0, line, pname );
		table.setEntry( 1, line, f->getPosFormatted( object_len[i], (object_speed[i] < 0 ), DEG_PRECISION_MORE ));
		if ( i != OTRUENODE && i != OMEANNODE ) table.setEntry( 2, line, f->getLatitudeFormated( object_lat[i], DEG_PRECISION_MORE ));
		line++;
	}
	table.setEntry( 0, line, _( "Ascendant" ));
	table.setEntry( 1, line, f->getPosFormatted( getHouse( 0, false, false), false, DEG_PRECISION_MORE ));
	line++;
	table.setEntry( 0, line, _( "Medium Coeli" ));
	table.setEntry( 1, line, f->getPosFormatted( getHouse( 9, false, false), false, DEG_PRECISION_MORE ) );
	writer->writeTable( table ); 

	Table t2( 2, 5 );
	t2.setHeader( 0,  _( "Name" ));
	t2.setHeader( 1,  _( "Value" ));
  writer->writeHeader2( _( "Ayanamsa" ));
	line = 1;
	for( int i = 0; i < NB_AYANAMSAS; i++ )
	{
		t2.setEntry( 0, line, f->getAyanamsaName( i+1 ));
		t2.setEntry( 1, line, f->getDegreesFormated( calculator->calcAyanamsa( getJD(), i+1 ), DEG_PRECISION_MORE ));
		line++;
	}
	t2.setEntry( 0, 4, _( "Custom" ) );
	t2.setEntry( 1, 4, f->getDegreesFormated( calculator->calcAyanamsa( getJD(), 4 ), DEG_PRECISION_MORE ));
	writer->writeTable( t2 ); 
}

/*****************************************************
**
**   SimpleHoroscope   ---   writeBhavas 
**
******************************************************/
void SimpleHoroscope::writeBhavas( Writer *writer, const bool &vedic )
{
	int i;
	Formatter *formatter = Formatter::get();
	Lang *lang = Lang::get();
	wxString s;

  writer->writeHeader1( _( "Bhavas" ));
	Table table( 3, 13 );
	table.setHeader( 0,  _( "Bhava" ));
	table.setHeader( 1,  _( "Cusp" ));
	table.setHeader( 2,  _( "Sandhi" ));

	int line = 1;
	for( i = HOUSE1; i <= HOUSE12; i++ )
	{
		//s.Printf( wxT( "%02d" ), i+1 );
		table.setEntry( 0, line, lang->getBhavaName( i ));
		table.setEntry( 1, line, formatter->getPosFormatted( getHouse(i, true, false), false ));
		table.setEntry( 2, line, formatter->getPosFormatted( getHouse(i, true, true), false ));
		line++;
	}

	writer->writeTable( table ); 

  writer->writeHeader2( _( "Planets in Bhavas" ));
	Table t2( 3, 11 );
	t2.setHeader( 0,  _( "Planet" ));
	t2.setHeader( 1,  _( "Length" ));
	t2.setHeader( 2,  _( "Bhava" ));
	line = 1;
	for( i = ISUN; i <= IASCENDANT; i++ )
	{
		t2.setEntry( 0, line, lang->getObjectName( i, TMEDIUM ) );
		t2.setEntry( 1, line, formatter->getPosFormatted( getObjectLength(i), isObjectRetrograde(i)));
		t2.setEntry( 2, line, lang->getBhavaName( getHousePos(i)));
		line++;
	}
	writer->writeTable( t2 ); 
}


