/*

	This is the source code of

   	M A I T R E Y A
    ===============

	Open source platform for Vedic and western astrology

  File           TextHelper.cpp
  Release        4.1
  Author         Martin Pettau
  Copyright (C)  2003-2006 by the author

  Released under the Artistic License as published by the
  Free Software Foundation, read the file 'COPYING' for more information.

*/

#ifdef __GNUG__
	#pragma implementation "TextHelper.h"
#endif

#include "TextHelper.h"

#include <wx/log.h>
#include <wx/string.h>
#include <math.h>

#include "Ashtakavarga.h"
#include "func.h"
#include "constants.h"
#include "IPlanet.h"
#include "Horoscope.h"
#include "Jaimini.h"
#include "Lang.h"
#include "constants.h"
#include "Conf.h"
#include "Varga.h"
#include "Writer.h"

extern Config *config;

/*****************************************************
**
**   TextHelper   ---   listPlanets
**
******************************************************/
void TextHelper::listPlanets( Horoscope *h, Writer *writer )
{
	if ( config->preferVedic ) listVedicPlanets( h, writer, config->iMainChildStyle );
	else listWesternPlanets( h, writer, config->wMainChildStyle );
}

/*****************************************************
**
**   TextHelper   ---   listVedicPlanets
**
******************************************************/
void TextHelper::listVedicPlanets( Horoscope *h, Writer *writer, const int &istyle )
{
	Lang *lang = Lang::get();
	Formatter *formatter = Formatter::get();
  wxChar tmp[256];
  int pindex, rowindex = 1, shast, col = 1;
  double len;
  bool retro;

  int nakshatramode =  ( config->iNakshatraMode28 ? N28 : N27 );
  VargaView view( h, 0 );
  VargaView bview( h, 18 );
  JaiminiExpert expert( &view );
  expert.calcCharaKarakas();
  AshtakavargaExpert aexpert( &view );
  aexpert.update();
  IPlanet *planet;

  int numcols, numrows, i = 1;
  int opt_nav, opt_kara, opt_shast, opt_nak, opt_dasa, opt_bha, opt_kp, opt_ashtaka, opt_pada, opt_whousepos;

  opt_nav = opt_kara = opt_shast = opt_nak = opt_dasa = opt_bha = opt_kp = opt_ashtaka = opt_pada = opt_whousepos = 0;

	numcols = 2;
	numrows = 11;
	if ( istyle & IMAIN_CHILD_SHOW_NAVAMSA) { opt_nav = true; numcols++; }
	if ( istyle & IMAIN_CHILD_SHOW_KARAKA) { opt_kara = true; numcols++; }
	if ( istyle & IMAIN_CHILD_SHOW_SHASTIAMSA) { opt_shast = true; numcols++; }
	if ( istyle & IMAIN_CHILD_SHOW_NAKSHATRA) { opt_nak = true; numcols++; }
	if ( istyle & IMAIN_CHILD_SHOW_DASAVARGA) { opt_dasa = true; numcols++; }
	if ( istyle & IMAIN_CHILD_SHOW_BHAVA) { opt_bha = true; numcols++; }
	if ( istyle & IMAIN_CHILD_SHOW_ASHTAKA ) { opt_ashtaka = true; numcols += 1; }
	if ( istyle & IMAIN_CHILD_SHOW_PADA) { opt_pada = true; numcols += 1; }
	if ( istyle & IMAIN_CHILD_SHOW_KP) { opt_kp = true; numcols += 2; }
	if ( istyle & IMAIN_CHILD_SHOW_LAGNAS) { numrows += 3; }
	if ( istyle & IMAIN_CHILD_SHOW_UPA) { numrows += 10; }
	if ( istyle & IMAIN_CHILD_SHOW_HOUSES ) { numrows += 11; }

  Table table( numcols, numrows );
  table.setHeader( 0,  _( "Planet" ));
  table.setHeader( 1,  _( "Length" ));
  i = 2;
	if ( opt_nav ) table.setHeader( i++, _( "Navamsa" ));
	if ( opt_nak ) table.setHeader( i++, _( "Nakshatra" ));
	if ( opt_pada ) table.setHeader( i++, _( "Pada" ));
	if ( opt_kara ) table.setHeader( i++, _( "Karaka" ));
	if ( opt_kp )
	{
		table.setHeader( i++, _( "KP Lord" ));
		table.setHeader( i++, _( "Sublord" ));
	}
	if ( opt_shast ) table.setHeader( i++, _( "Shastiamsa" ));
	if ( opt_dasa ) table.setHeader( i++, _( "Dasa Varga" ));
	if ( opt_bha ) table.setHeader( i++, _( "Bhava" ));
	if ( opt_ashtaka ) table.setHeader( i++, _( "AV" ));

  for( pindex = ISUN; pindex < MAX_IOBJECTS_LAGNA_UPA + 11; pindex++ )
  {
    // Loop conditions
    if ( pindex > IASCENDANT && pindex < MAX_IOBJECTS_LAGNA && ! ( istyle & IMAIN_CHILD_SHOW_LAGNAS )) continue;
    if ( pindex >= MAX_IOBJECTS_LAGNA && pindex < MAX_IOBJECTS_LAGNA_UPA
			&& ! ( istyle & IMAIN_CHILD_SHOW_UPA )) continue;
    if ( pindex >= MAX_IOBJECTS_LAGNA_UPA && ! ( istyle & IMAIN_CHILD_SHOW_HOUSES )) continue;

    retro = false;
    planet = 0;
    col = 2;
    assert( (int)table.nb_rows >= rowindex );
    if ( pindex >=  MAX_IOBJECTS_LAGNA_UPA )
    {
      len = h->getHouse( pindex - MAX_IOBJECTS_LAGNA_UPA+1, true, false );
      //wxSprintf( tmp, wxT( "%02d" ), pindex - MAX_IOBJECTS_LAGNA_UPA + 2 );
			table.setEntry( 0, rowindex, lang->getBhavaName( pindex - MAX_IOBJECTS_LAGNA_UPA + 1 ));
    }
    else
    {
      len = h->getObjectLength( pindex );
			table.setEntry( 0, rowindex, lang->getObjectName( pindex, TLARGE ) );
      if ( pindex <= ISATURN ) planet = h->getIPlanet( pindex );
    }
    if ( pindex >= IMARS && pindex <= ISATURN && h->isObjectRetrograde( pindex ) ) retro = true;
		table.setEntry( 1, rowindex, formatter->getPosFormatted( len, retro ));

    // Navamsa
    if ( istyle & IMAIN_CHILD_SHOW_NAVAMSA)
			table.setEntry( col++, rowindex, lang->getSignName( getRasi( len * 9 ), config->signPrecision ));

    // Nakshatra
    if ( istyle & IMAIN_CHILD_SHOW_NAKSHATRA)
      table.setEntry( col++, rowindex,
				lang->getNakshatraName( getNakshatra( len, nakshatramode ), nakshatramode, TMEDIUM ));

    // Pada
    if ( istyle & IMAIN_CHILD_SHOW_PADA)
    {
      wxSprintf( tmp, wxT( "%d" ), (int)(getNakshatraLength( len, nakshatramode ) / 3.3333333333 ) + 1 );
      table.setEntry( col++, rowindex, tmp );
    }

    // Karaka
    if ( istyle & IMAIN_CHILD_SHOW_KARAKA)
    {
      if ( pindex <= IRAHU )
				table.setEntry( col, rowindex, lang->getKarakaName( expert.getCharaKarakaProperty( pindex )) );
      col++;
   }

    // Krishnamurti
    if ( istyle & IMAIN_CHILD_SHOW_KP)
    {
			h->updateKP( 0 );
      table.setEntry( col++, rowindex, lang->getObjectName( h->getKPLord( pindex, false ), TLARGE )  );
      table.setEntry( col++, rowindex, lang->getObjectName( h->getKPLord( pindex, true ), TLARGE )  );
    }

    // Shastiamsa
    if ( istyle & IMAIN_CHILD_SHOW_SHASTIAMSA)
    {
      shast = (int)( a_red( len, 30 ) * 2 );
      if ( isEvenRasi( len )) shast = 59 - shast;
      assert( shast >= 0 && shast < 60 );
      wxSprintf( tmp, wxT( "%s (%c)" ), lang->getShastiamsaName( shast).c_str(),
				( k_shastiamsa_benefic[(int)shast] ? 'B' : 'M' ));
      table.setEntry( col, rowindex, tmp );
      col++;
    }

    // Dasavarga Dignity
    if ( istyle & IMAIN_CHILD_SHOW_DASAVARGA)
    {
      if ( pindex <= ISATURN )
			{
				VargaExpert expert;
				table.setEntry( col, rowindex, expert.getDasamsAvastha((int)planet->getDasamsaAvastha()) );
			}
      col++;
    }

    // Bhava
    if ( istyle & IMAIN_CHILD_SHOW_BHAVA)
    {
      if ( pindex <= IKETU )
      {
        wxSprintf( tmp, wxT( "%d" ), red12( bview.getRasi( pindex ) - bview.getRasi( IASCENDANT )) + 1 );
        table.setEntry( col++, rowindex, tmp );
      }
    }

    // Ashataka Varga
    if ( istyle & IMAIN_CHILD_SHOW_ASHTAKA)
    {
      if (( pindex >= ISUN && pindex <= ISATURN ) || pindex == IASCENDANT )
      {
        wxSprintf( tmp, wxT( "%d" ), aexpert.getItem( REKHA, pindex, getRasi( len )));
        table.setEntry( col++, rowindex, tmp );
      }
      col++;
    }
    rowindex++;
  }
	writer->writeTable( table ); 
}

/*****************************************************
**
**   TextHelper   ---   listWesternPlanets
**
******************************************************/
void TextHelper::listWesternPlanets( Horoscope *h, Writer *writer, const int &wstyle )
{
  Lang *lang = Lang::get();
  Formatter *formatter = Formatter::get();
  wxChar tmp[256];
  int pindex, j, maxrow, currentrow;
  int numcols, numrows;
  int opt_nav, opt_kara, opt_shast, opt_nak, opt_dasa, opt_bha, opt_kp, opt_ashtaka, opt_pada, opt_whousepos;

  opt_nav = opt_kara = opt_shast = opt_nak = opt_dasa = opt_bha = opt_kp = opt_ashtaka = opt_pada = opt_whousepos = 0;

	numcols = 2;
	numrows = 14;
	if ( wstyle & WMAIN_CHILD_SHOW_HOUSEPOS ) { opt_whousepos = true; numcols++; }
	if ( wstyle & WMAIN_CHILD_SHOW_HOUSES ) { numrows += 4; }
	if ( wstyle & WMAIN_CHILD_SHOW_URANIAN ) { numrows += 8; }
  Table table( numcols, numrows );
  table.setHeader( 0,  _( "Planet" ));
  table.setHeader( 1,  _( "Length" ));
	if ( opt_whousepos ) table.setHeader( 2, _( "House" ));


	maxrow = 12;
  if ( wstyle & WMAIN_CHILD_SHOW_URANIAN ) maxrow+= 8;
  
  for( j = 0; j <= maxrow; j++ )
  {
    pindex = j + WSUN;
    assert( (int)table.nb_rows > j );
    table.setEntry( 0, j+1, lang->getObjectName( pindex, TLARGE ));
    table.setEntry( 1, j+1, formatter->getPosFormatted( h->getObjectLength(pindex), h->isObjectRetrograde(pindex)) );
    if ( wstyle & WMAIN_CHILD_SHOW_HOUSEPOS )
    {
      assert( table.nb_cols > 0 );
      wxSprintf( tmp, wxT( "%d" ), h->getHousePos( pindex ) + 1 );
			table.setEntry( 2, j+1, tmp );
    }
  }
  if ( wstyle & WMAIN_CHILD_SHOW_HOUSES )
  { 
    currentrow = maxrow+1;
    for( j = 0; j < 12; j++ )
    {
      if (( j == HOUSE1 ) || ( j == HOUSE4 ) || ( j == HOUSE5 ) || ( j == HOUSE6 )
        || ( j == HOUSE7 ) || ( j == HOUSE8 ) || ( j == HOUSE9 ) || ( j == HOUSE10 )
      ) continue;
      assert( (int)table.nb_rows > currentrow );
      wxSprintf( tmp, wxT( "%02d" ), j+1 );
			table.setEntry( 0, currentrow+1, tmp );
			table.setEntry( 1, currentrow+1, formatter->getPosFormatted( h->getHouse(j, false), false ));
      currentrow++;
    }
  }

	writer->writeTable( table ); 
}


/*****************************************************
**
**   TextHelper   ---   writeBaseData 
**
******************************************************/
void TextHelper::writeBaseData( Horoscope *h, Writer *writer )
{
	wxString s;
	Formatter *f = Formatter::get();
	double time;

	DataSet *ds = h->getDataSet();
	Location *loc = ds->getLocation();
	JDate *date = ds->getDate();

  writer->writeHeader1( _( "Base Data" ));
	Table table( 2, 17 );
	/*
	Table table( 2, 18 );
	table.setHeader( 0,  _( "Item" ));
	table.setHeader( 1,  _( "Value" ));
	*/

	int line = 0;
	table.setEntry( 0, line,  _( "Name" ));
	table.setEntry( 1, line++, h->getHName() );

	table.setEntry( 0, line,  _( "Date" ));
	s << f->getDateStringFromJD( h->getJD() + ( loc->getTimeZone() + loc->getDST() )/24 )
		<< wxT( " " ) << Lang::get()->getWeekdayName(date->getWeekDay());
	table.setEntry( 1, line++, s );

	table.setEntry( 0, line,  _( "Local Time" ));
  time = a_red( getTimeFromJD( date->getJD()) + loc->getTimeZone() + loc->getDST(), 24 );
	table.setEntry( 1, line++, f->getTimeFormated( time ));

	table.setEntry( 0, line,  _( "Location" ));
	table.setEntry( 1, line++, loc->getLocName() );

	table.setEntry( 0, line,  _( "Longitude" ));
	table.setEntry( 1, line++, f->getLongitudeFormated( loc->getLongitude()));

	table.setEntry( 0, line,  _( "Latitude" ));
	table.setEntry( 1, line++, f->getLatitudeFormated( loc->getLatitude()));

	table.setEntry( 0, line,  _( "Time Zone" ));
	double tz = loc->getTimeZone();
  s.Printf( wxT( "%s %c%.1f %s" ), _( "UT" ), ( tz >= 0 ? '+' : '-' ), tz, _( "Hours" ));
	table.setEntry( 1, line++, s );

	table.setEntry( 0, line,  _( "Daylight Saving" ));
  s.Printf( wxT( "%.1f %s" ), loc->getDST(), _( "Hours" ));
	table.setEntry( 1, line++, s );

	table.setEntry( 0, line,  _( "Universal Time" ));
	table.setEntry( 1, line++, f->getTimeFormated( getTimeFromJD( h->getJD() )));

	table.setEntry( 0, line,  _( "Sidereal Time" ));
	table.setEntry( 1, line++, f->getTimeFormated( h->getSiderealTime()));

	table.setEntry( 0, line,  _( "Julian Date" ));
  s.Printf( wxT( "%8.5f" ), h->getJD());
	table.setEntry( 1, line++, s );

	table.setEntry( 0, line,  _( "Vedic Ayanamsa" ));
	table.setEntry( 1, line++, f->getAyanamsaName( h->getIAyanamsaType() ));

	table.setEntry( 0, line,  _( "Ayanamsa Value" ));
	table.setEntry( 1, line++, f->getDegreesFormated( h->getIAyanamsa()));

	table.setEntry( 0, line,  _( "Western Ayanamsa" ));
	table.setEntry( 1, line++, f->getAyanamsaName( h->getWAyanamsaType()));

	table.setEntry( 0, line,  _( "Ayanamsa Value" ));
	table.setEntry( 1, line++, f->getDegreesFormated( h->getWAyanamsa()));

	table.setEntry( 0, line,  _( "Sunrise" ));
	table.setEntry( 1, line++, f->getTimeFormated( getTimeFromJD( h->getSunrise() + ( loc->getTimeZone() + loc->getDST() ) / 24.0 )) );

	table.setEntry( 0, line,  _( "Sunset" ));
	table.setEntry( 1, line++, f->getTimeFormated( getTimeFromJD( h-> getSunset() + ( loc->getTimeZone() + loc->getDST() ) / 24.0 )) );

	writer->writeTable( table ); 
}

