/*

	This is the source code of

   	M A I T R E Y A
    ===============

	Open source platform for Vedic and western astrology

  File           Varga.cpp
  Release        4.1
  Author         Martin Pettau
  Copyright (C)  2003-2006 by the author

  Released under the Artistic License as published by the
  Free Software Foundation, read the file 'COPYING' for more information.

*/

#ifdef __GNUG__
	#pragma implementation "Varga.h"
#endif

#include "Varga.h"

#include <wx/log.h>
#include <wx/string.h>
#include <math.h>

#include "Conf.h"
#include "constants.h"
#include "func.h"
#include "Horoscope.h"
#include "IPlanet.h"
#include "Lang.h"
#include "Writer.h"

extern Config *config;


const int VargaExpert::vargaDivisions[NB_VARGAS] =
	{ 1, 9, 2, 3, 4, 6, 7, 8, 10, 12, 16, 20, 24, 27, 30, 40, 45, 60 }; //, -1 };

/*****************************************************
**
**   Varga   ---   Constructor 
**
******************************************************/
Varga::Varga( const int& i, const int &d )
{
	index = i;
	division = d;
	rasi = 0;
	bala = length = 0;
}

/*****************************************************
**
**   Expert   ---   Constructor 
**
******************************************************/
Expert::Expert()
{
	view = 0;
	horoscope = 0;
}

/*****************************************************
**
**   Expert   ---   Constructor 
**
******************************************************/
Expert::Expert( Horoscope *h )
 : horoscope( h )
{
	view = 0;
}

/*****************************************************
**
**   Expert   ---   Constructor 
**
******************************************************/
Expert::Expert( VargaView *v )
 : view( v )
{
	horoscope = 0;
}

/*****************************************************
**
**   VargaExpert   ---   getNbVargas 
**
******************************************************/
int VargaExpert::getNbVargas()
{
	return NB_VARGAS;
}

/**************************************************************
***
**  VargaExpert   ---   getVargaName
***
***************************************************************/
const wxChar *VargaExpert::getVargaName( const int &i )
{
	const wxChar *vargaNames[NB_VARGAS] =
	{
		_( "Rasi" ), _( "Navamsa" ), _( "Hora" ), _( "Drekkana" ), _( "Chaturtamsa" ), _( "Sastamsa" ),
		_( "Saptamamsa" ), _( "Ashtamsa" ), _( "Dasamsa" ), _( "Dvadasamsa" ), _( "Shodasamsa" ),
		_( "Vimsamsa" ), _( "Siddhamsa" ), _( "Bhamsa" ), _( "Trimsamsa" ),
		_( "Chatvarimsamsa" ), _( "Akshavedamsa" ), _( "Shastiamsa" ), _( "Bhava" ) 
	};
	if (( i < 0 ) || ( i >= NB_VARGAS ))
	{
		wxLogError( wxT( "ERROR: no varga possible for index %d, exiting" ), i );
		exit( 1 );
	}
	return vargaNames[i];
}

/*****************************************************
**
**   VargaExpert   ---   getVargaIndexByDivision 
**
******************************************************/
int VargaExpert::getVargaIndexByDivision( const int &div )
{
	int j = -1;

	assert(( div >= 0 ) || (div <= 60 ));
	while( ++j < NB_VARGAS ) if ( vargaDivisions[j] == div ) return j;

	wxLogError( wxT( "ERROR (VargaExpert::getVargaIndexByDivision) --- No vargas found for division %d" ), div );
	return 0;
}

/*****************************************************
**
**   <-- VargaExpert   ---   getVargaDivision -->
**
******************************************************/
const int VargaExpert::getVargaDivision( const int &i )
{
	return vargaDivisions[i];
}

/*****************************************************
**
**   VargaExpert   ---   calcVarga 
**
******************************************************/
void VargaExpert::calcVarga( Varga *varga, const double& l, const int &division )
{
	double ret = 0;
	double len = l;
	double basepos, rs;
	int i, r;
	double hstart, hend;
	assert( horoscope );

	switch ( division )
	{
		// D1
		case 0:
			ret = len;
		break;

		// D9
		case 1:
			ret =  9 * len;
		break;

		// D2
		case 2:
			switch( config->iHoraMode )
			{
				case HORA_PARASARA:
					ret = a_red( len - 15, 60 ) + 90;
				break;
				case HORA_JAGANNATH:
					rs = getRasiLen( len );
					if ((( isOddRasi( len )) && ( rs <= 15 )) || (( isEvenRasi( len )) && ( rs > 15 )))
						ret = len;
					else ret = len + 180;
				break;
				case HORA_PARIVRITTIDVAYA:
					ret = 2 * len;
				break;
				default:
					assert( false );
				break;
			}
		break;

		// D3
		case 3:
			switch( config->iDrekkanaMode )
			{
				case DREKKANA_PARASARA:
					ret = ( floor( getRasiLen( len ) / 10 )  * 120 + getRasi( len ) * 30 + getRasiLen( 3 * len ));
				break;
				case DREKKANA_JAGANNATH:
					len = a_red( len, 120 );
					ret = ( floor( getRasiLen( len ) / 10 )  * 120 + getRasi( len ) * 30 + getRasiLen( 3 * len ));
				break;
				case DREKKANA_PARIVRITTITRAYA:
					ret = 3 * len;
				break;
				case DREKKANA_SOMNATH:
					if ( isOddRasi( len ))
					{
						r = (int)(getRasi( len ) / 2 );
						len -= 30 * r;
						ret = 3 * len;
					}
					else
					{
						len -= 30;
						r = (int)(getRasi( len ) / 2 );
						len -= 30 * r;
						ret = 360 - ( 3 * len );
					}
				break;
				default:
					assert( false );
				break;
			}
		break;

		// D4
		case 4:
			ret = ( floor( getRasiLen( len ) / 7.5 )  * 90 + getRasi( len ) * 30 + getRasiLen( 4 * len ));
		break;

		// D6
		case 5:
			ret =  6 * len;
			//ret = ( getRasi( len ) * 30 + getRasiLen( len ) * 5 );
		break;

		// D7
		case 6:
			r = getRasi( len );
			basepos = getRasi( len ) * 30 + getRasiLen( len ) * 7;
			if ( isOddRasi( len ) == 1 ) ret = basepos;
			else ret = basepos + 180;
		break;

		// D8
		case 7:
			ret = 8 * len;
		break;

		// D10
		case 8:
			basepos = getRasi( len ) * 30 + getRasiLen( len ) * 10;
			if ( isOddRasi( len ) == 1 ) ret = basepos;
			else ret = basepos + 240;
		break;

		// D12
		case 9:
			ret = ( getRasi( len ) * 30 + getRasiLen( len ) * 12 );
		break;

		// D16
		case 10:
			ret = 16 * len;
		break;

		// D20
		case 11:
			ret = 20 * len;
		break;

		// D24
		case 12:
			basepos = getRasiLen( len ) * 24;
			if ( isOddRasi( len ) == 1 ) ret = basepos + 120;
			else ret = basepos + 90;
		break;

		// D27
		case 13:
			ret =  27 * len;
		break;

		// D30
		case 14:
			rs = getRasiLen( len );
			if ( isOddRasi( len ) == 1 )
			{
				// Ar, Aq, Sa, Ge, Li
				if ( rs < 5 )
					ret = 30 * ARIES + rs*6;
				else if (( rs >=  5 ) && ( rs <= 10 ))
					ret = 30 * AQUARIUS + (rs - 5)*6;
				else if (( rs >= 10 ) && ( rs <= 18 ))
					ret = 30 * SAGITTARIUS + (rs - 10)/4 * 15;
				else if (( rs >=  18 ) && ( rs <= 25 ))
					ret = 30 * GEMINI + ( rs - 18 )/7 * 30;
				else if ( rs > 25 )
					ret = 30 * LIBRA + (rs - 25)*6;
			}
			else
			{
				// Ta, Vi, Pi, Cp, Sc
				if ( rs < 5 ) ret = 30 * TAURUS + (5-rs)*6;
				else if (( rs >=  5 ) && ( rs <= 10 )) ret = 30 * VIRGO + (10-rs)*6;
				else if (( rs >= 10 ) && ( rs <= 18 )) ret = 30 * PISCES + ( 18-rs)/4 * 15;
				else if (( rs >= 18 ) && ( rs <= 25 )) ret = 30 * CAPRICORNUS + (25-rs)/7 * 30;
				else if ( rs > 25 ) ret = 30 * SCORPIO + ( 30 -rs)*6;
			}
		break;

		// D40
		case 15:
			basepos = getRasiLen( len ) * 40;
			if ( isOddRasi( len ) == 1 ) ret = basepos;
			else ret = basepos + 180;
		break;

		// D45
		case 16:
			basepos = getRasiLen( len ) * 45;
			if ( inMovableSign( len )) ret = basepos;
			else if ( inFixedSign( len )) ret = basepos + 120;
			else ret = basepos + 240;
		break;

		// D60
		case 17:
			ret = 60 * getRasiLen( len ) + getRasi( len ) * 30;
		break;

		// Bhava
		case 18:
			for( i = HOUSE1; i <= HOUSE12; i++ )
			{
				assert( horoscope );
				if ( config->iHouseWestern )
				{
					hstart = horoscope->getHouse( i, true, false );
					hend = horoscope->getHouse( red12(i+1), true, false );
				}
				else
				{
					hstart = horoscope->getHouse( red12(i-1), true, true );
					hend = horoscope->getHouse( i, true, true );
				}
				if (( hstart <= len && len < hend ) || ( hstart > hend && ( len >= hstart || len < hend )))
				{
					ret = red_deg( 30 * ( i + getRasi( horoscope->getObjectLength( IASCENDANT ))) + 15 );
					break;
				}
			}
		break;

		default:
			assert( 0 );
		break;
	}
	ret = red_deg( ret );
	varga->length = ret;
	varga->rasi = red12( (int)(ret/30) );
}

/*****************************************************
**
**   VargaExpert   ---   calcBala 
**
******************************************************/
void VargaExpert::calcBala( const int &pindex, const double &len, const int &division )
{
	double bala = 0;
	int rasi = getRasi( len );
	assert( horoscope );
	int istempfriend = isRasiPosTempFriend( rasi, horoscope->getVarga( getLord( rasi), division )->getRasi());
	int isfriend = isFriend( pindex, getLord( rasi));

	assert(( pindex >= ISUN ) && ( pindex <= IKETU ));

	//if ( division != 2 )
	if ( 1 )
	{
		if ( hasExaltationRasi( pindex, rasi )) bala = 20;

		else if ( hasDebilationRasi( pindex, rasi )) bala = 0;

		else if ( isInMulatrikona( pindex, len )) bala = 20;

		else if ( pindex == getLord( rasi )) bala = 20;

		// Adhi mitra
		else if ( isfriend == FRIEND && istempfriend == FRIEND ) bala = 18;

		// mitra
		else if ( isfriend == NEUTRAL && istempfriend == FRIEND ) bala = 15;

		// sama
		else if (( isfriend == FRIEND && istempfriend == ENEMY ) || ( isfriend == ENEMY && istempfriend == FRIEND )) bala = 10;

		// satru
		else if ( isfriend == NEUTRAL && istempfriend == ENEMY ) bala = 7;

		// adhi satru
		else if ( isfriend == ENEMY && istempfriend == ENEMY ) bala = 5;
		else assert( 0 );

		/*
		if ( hasExaltationRasi( pindex, rasi )) bala = 20;
		else if ( hasDebilationRasi( pindex, rasi )) bala = 0;
		else if ( isInMulatrikona( pindex, len )) bala = 15;
		else if ( pindex == getLord( rasi )) bala = 10;
		else if ( isFriend( pindex, getLord( rasi)) == FRIEND ) bala = 5;
		else if ( isFriend( pindex, getLord( rasi)) == NEUTRAL ) bala = 2.5;
		else if ( isFriend( pindex, getLord( rasi)) == ENEMY ) bala = 1.25;
		*/
	}
	else //( division == 2 ) ---  Hora
	{
		if (( pindex == ISUN ) || ( pindex == IMARS ) || ( pindex == IJUPITER ))
		{
			if ( rasi == LEO ) bala = 20;
			else bala = 0;
		}
		else if (( pindex == IMOON ) || ( pindex == IVENUS ) || ( pindex == ISATURN ))
		{
			if ( rasi == CANCER ) bala = 20;
			else bala = 0;
		}
		else if ( pindex == IMERCURY )
		{
			bala = 20;
		}
		else if (( pindex == IRAHU ) || ( pindex == IKETU ))
		{
			bala = 0;
		}
		else
		{
			wxLogError( wxT( "ERROR invalid planet for hora %s" ),  Lang::get()->getObjectName( pindex, TMEDIUM ) );
			exit(1);
		}
	}
	horoscope->getVarga( pindex, division )->bala = bala;
}


/*****************************************************
**
**   VargaExpert   ---   calcShadVargas 
**
******************************************************/
void VargaExpert::calcShadVargas( Horoscope *horoscope, const int &pl, double &shadvarga, double &saptavarga, double &dasavarga,
		double &shodasavarga, double &dasamsa_avastha )
{
	shadvarga =
		  6 * horoscope->getVarga( pl, getVargaIndexByDivision(1))->getBala()
		+ 2 * horoscope->getVarga( pl, getVargaIndexByDivision(2))->getBala()
		+ 4 * horoscope->getVarga( pl, getVargaIndexByDivision(3))->getBala()
		+ 5 * horoscope->getVarga( pl, getVargaIndexByDivision(9))->getBala()
		+ 2 * horoscope->getVarga( pl, getVargaIndexByDivision(12))->getBala()
		+ 1 * horoscope->getVarga( pl, getVargaIndexByDivision(30))->getBala()
	;
	shadvarga /= 20;

	saptavarga =
		  5 * horoscope->getVarga( pl, getVargaIndexByDivision(1))->getBala()
		+ 2 * horoscope->getVarga( pl, getVargaIndexByDivision(2))->getBala()
		+ 3 * horoscope->getVarga( pl, getVargaIndexByDivision(3))->getBala()
		+ 2.5 * horoscope->getVarga( pl, getVargaIndexByDivision(9))->getBala()
		+ 4.5 * horoscope->getVarga( pl, getVargaIndexByDivision(12))->getBala()
		+ 2 * horoscope->getVarga( pl, getVargaIndexByDivision(30))->getBala()
		+ 1 * horoscope->getVarga( pl, getVargaIndexByDivision(7))->getBala()
	;
	saptavarga /= 20;

	dasavarga =
		  3 * horoscope->getVarga( pl, getVargaIndexByDivision(1))->getBala()
		+ 1.5 * horoscope->getVarga( pl, getVargaIndexByDivision(2))->getBala()
		+ 1.5 * horoscope->getVarga( pl, getVargaIndexByDivision(3))->getBala()
		+ 1.5 * horoscope->getVarga( pl, getVargaIndexByDivision(9))->getBala()
		+ 1.5 * horoscope->getVarga( pl, getVargaIndexByDivision(12))->getBala()
		+ 1.5 * horoscope->getVarga( pl, getVargaIndexByDivision(30))->getBala()
		+ 1.5 * horoscope->getVarga( pl, getVargaIndexByDivision(7))->getBala()
		+ 1.5 * horoscope->getVarga( pl, getVargaIndexByDivision(10))->getBala()
		+ 1.5 * horoscope->getVarga( pl, getVargaIndexByDivision(16))->getBala() // Kalamsa
		+ 5 * horoscope->getVarga( pl, getVargaIndexByDivision(60))->getBala()
	;
	dasavarga /= 20;

	dasamsa_avastha = 0;
	if ( horoscope->getVarga( pl, getVargaIndexByDivision(1))->getBala() >= 10 ) dasamsa_avastha++;
	if ( horoscope->getVarga( pl, getVargaIndexByDivision(2))->getBala() >= 10 ) dasamsa_avastha++;
	if ( horoscope->getVarga( pl, getVargaIndexByDivision(3))->getBala() >= 10 ) dasamsa_avastha++;
	if ( horoscope->getVarga( pl, getVargaIndexByDivision(9))->getBala() >= 10 ) dasamsa_avastha++;
	if ( horoscope->getVarga( pl, getVargaIndexByDivision(12))->getBala() >= 10 ) dasamsa_avastha++;
	if ( horoscope->getVarga( pl, getVargaIndexByDivision(30))->getBala() >= 10 ) dasamsa_avastha++;
	if ( horoscope->getVarga( pl, getVargaIndexByDivision(7))->getBala() >= 10 ) dasamsa_avastha++;
	if ( horoscope->getVarga( pl, getVargaIndexByDivision(10))->getBala() >= 10 ) dasamsa_avastha++;
	if ( horoscope->getVarga( pl, getVargaIndexByDivision(16))->getBala() >= 10 ) dasamsa_avastha++;

	// Shastiamsa
	double len = horoscope->getVarga( pl, 0 )->getLength();
	int s = (int)( 2 * a_red( len, 30 ));
	if ( isEvenRasi( len )) s = 60 - s;
	if (( pl <= ISATURN)&&( k_shastiamsa_benefic[s] )) dasamsa_avastha++;

	shodasavarga =
		  3.5 * horoscope->getVarga( pl, getVargaIndexByDivision(1))->getBala()
		+ 1 * horoscope->getVarga( pl, getVargaIndexByDivision(2))->getBala()
		+ 1 * horoscope->getVarga( pl, getVargaIndexByDivision(3))->getBala()
		+ 3 * horoscope->getVarga( pl, getVargaIndexByDivision(9))->getBala()
		+ .5 * horoscope->getVarga( pl, getVargaIndexByDivision(12))->getBala()
		+ 1 * horoscope->getVarga( pl, getVargaIndexByDivision(30))->getBala()
		+ .5 * horoscope->getVarga( pl, getVargaIndexByDivision(7))->getBala()
		+ .5 * horoscope->getVarga( pl, getVargaIndexByDivision(10))->getBala()
		+ 2 * horoscope->getVarga( pl, getVargaIndexByDivision(16))->getBala() // Kalamsa
		+ 4 * horoscope->getVarga( pl, getVargaIndexByDivision(60))->getBala()
		+ .5 * horoscope->getVarga( pl, getVargaIndexByDivision(20))->getBala()
		+ .5 * horoscope->getVarga( pl, getVargaIndexByDivision(24))->getBala()
		+ .5 * horoscope->getVarga( pl, getVargaIndexByDivision(27))->getBala()
		+ .5 * horoscope->getVarga( pl, getVargaIndexByDivision(4))->getBala()
		+ .5 * horoscope->getVarga( pl, getVargaIndexByDivision(40))->getBala()
		+ .5 * horoscope->getVarga( pl, getVargaIndexByDivision(45))->getBala()
	;
	shodasavarga /= 20;
}

/*****************************************************
**
**   VargaExpert   ---   getDasamsAvastha
**
******************************************************/
const wxString VargaExpert::getDasamsAvastha( const int &i )
{
	assert( i >= 0 && i < 11 );
	const wxChar *das_avastha[11] = { _( " -" ), _( " -" ), _( "Parijata" ), _( "Uttama" ),
		_( "Gopura" ), _( "Simhasana" ), _( "Paravata" ), _( "Devaloka" ), _( "Brahmaloka" ),
		_( "Sakravahana" ), _( "Sridhama"  )}; 
	return das_avastha[i];
}

/*****************************************************
**
**   VargaExpert   ---   write
**
******************************************************/
void VargaExpert::write( Writer *writer )
{
	assert( horoscope );

	int p, v;
	double bala;
	wxChar tmp[256];
	wxString s;
	Lang *lang = Lang::get();
	IPlanet *planet;

	writer->writeHeader1( _( "Vargas" ));
	int num_varga = getNbVargas();

	Table t1( num_varga +2, 11 );
	t1.setHeader( 0, wxT( "" ));
	for( v = 0; v < num_varga-1; v++ )
	{
		s.Printf( wxT( "%02d" ), getVargaDivision(v));
		t1.setHeader( v+1, s );
	}
	t1.setHeader( num_varga, _( "Own" ));
	t1.setHeader( num_varga+1, _( "Loka" ));

	int line = 1;
	for( p = ISUN; p <= IASCENDANT; p++ )
	{
		t1.setEntry( 0, line, lang->getObjectName( p, TMEDIUM ));
		planet = horoscope->getIPlanet( p );

		for( v = 0; v < num_varga-1; v++ )
			t1.setEntry( v+1, line, Lang::get()->getSignName( horoscope->getVarga( p, v )->getRasi()));
		if ( p <= ISATURN )
		{
			wxSprintf( tmp, wxT( "%02.0f" ), planet->getDasamsaAvastha() );
			if ( tmp[0] == '0' ) tmp[0] = ' ';
			t1.setEntry( num_varga, line, tmp );
			t1.setEntry( num_varga+1, line, getDasamsAvastha((int)(planet->getDasamsaAvastha())) );
		}
		line++;
	}
	writer->writeTable( t1 );

	/*
	* Vimsopaka Strength
	*/
	writer->writeHeader2( _( "Vimsopaka Strength" ));
	Table t2( num_varga, 8 );
	t2.setHeader( 0, wxT( "" ));
	for( v = 0; v < num_varga-1; v++ )
	{
		s.Printf( wxT( "%02d" ), getVargaDivision(v) );
		t2.setHeader( v+1, s );
	}

	line = 1;
	for( p = 0; p <= ISATURN; p++ )
	{
		t2.setEntry( 0, line, Lang::get()->getObjectName( p, TMEDIUM ));
		planet = horoscope->getIPlanet( p );

		for( v = 0; v < num_varga-1; v++ )
		{
			bala = horoscope->getVarga( p, v )->getBala();
			wxSprintf( tmp, wxT( "%02.1f" ), bala );
			t2.setEntry( v+1, line, tmp );
		}
		line++;
	}
	writer->writeTable( t2 );

	writer->writeHeader2( _( "Shadvarga, etc" ));
	Table t3( 5, 8 );
	t3.setHeader( 0, wxT( "" ));
	t3.setHeader( 1, _( "Shadvarga" ));
	t3.setHeader( 2, _( "Saptavarga" ));
	t3.setHeader( 3, _( "Dasavarga" ));
	t3.setHeader( 4, _( "Shodasavarga" ));
	line = 1;
	for( p = 0; p <= ISATURN; p++ )
	{
		t3.setEntry( 0, line,  Lang::get()->getObjectName( p, TMEDIUM ));
		planet = horoscope->getIPlanet( p );
		wxSprintf( tmp, wxT( "%2.1f" ), planet->getShadVarga());
		t3.setEntry( 1,  line, tmp );

		wxSprintf( tmp, wxT( "%2.1f" ), planet->getSaptaVarga());
		t3.setEntry( 2,  line, tmp );

		wxSprintf( tmp, wxT( "%2.1f" ), planet->getDasaVarga());
		t3.setEntry( 3,  line, tmp );

		wxSprintf( tmp, wxT( "%2.1f" ), planet->getShodasaVarga());
		t3.setEntry( 4,  line, tmp );
		line++;
	}
	writer->writeTable( t3 );
}

/*****************************************************
**
**   VargaView   ---   Constructor 
**
******************************************************/
VargaView::VargaView( const int div )
{
	horoscope = 0;
	setDivision( div );
}

/*****************************************************
**
**   VargaView   ---   Constructor 
**
******************************************************/
VargaView::VargaView( Horoscope *h, const int div )
{
	horoscope = h;
	division = div;
}

/*****************************************************
**
**   VargaView   ---   Constructor 
**
******************************************************/
VargaView::VargaView( const VargaView &v )
{
	this->horoscope = v.horoscope;
	this->division = v.division;
}

/*****************************************************
**
**   VargaView   ---   setDivision 
**
******************************************************/
void VargaView::setDivision( const int &div )
{
	VargaExpert expert;
	assert( div >= 0 && div <= expert.getNbVargas() );
	division = div;
}

/*****************************************************
**
**   VargaView   ---   setHoroscope 
**
******************************************************/
void VargaView::setHoroscope( Horoscope *h )
{
	horoscope = h;
}

/*****************************************************
**
**   VargaView   ---   getLength 
**
******************************************************/
double VargaView::getLength( const int &planet )
{
	assert( planet >= 0 && planet < MAX_IOBJECTS_LAGNA_UPA && horoscope );
	return horoscope->getVarga( planet, division )->getLength();
}

/*****************************************************
**
**   VargaView   ---   getRasiLength 
**
******************************************************/
double VargaView::getRasiLength( const int &planet )
{
	assert( planet >= 0 && planet < MAX_IOBJECTS_LAGNA_UPA && horoscope );
	// reverse for rahu and ketu
	return getRasiLen( getLength( planet ), planet == IRAHU || planet == IKETU );
}

/*****************************************************
**
**   VargaView   ---   getRasi 
**
******************************************************/
int VargaView::getRasi( const int &planet )
{
	assert( planet >= 0 && planet < MAX_IOBJECTS_LAGNA_UPA && horoscope );
	return ::getRasi( getLength( planet ));
}

/*****************************************************
**
**   VargaView   ---   getBhava 
**
******************************************************/
int VargaView::getBhava( const int &planet )
{
	assert( planet >= 0 && planet < MAX_IOBJECTS_LAGNA_UPA && horoscope );
	return red12( this->getRasi( planet ) - this->getRasi( IASCENDANT ));
}

/*****************************************************
**
**   VargaView   ---   getNakshatra 
**
******************************************************/
int VargaView::getNakshatra( const int &planet, const int &nsys )
{
	assert( planet >= 0 && planet < MAX_IOBJECTS_LAGNA_UPA && horoscope );
	return ::getNakshatra( this->getLength( planet ), nsys );
}

/*****************************************************
**
**   VargaView   ---   getNakshatraLength 
**
******************************************************/
double VargaView::getNakshatraLength( const int &planet, const int &nsys )
{
	assert( planet >= 0 && planet < MAX_IOBJECTS_LAGNA_UPA && horoscope );
	return ::getNakshatraLength( this->getLength( planet ), nsys );
}

/*****************************************************
**
**   VargaView   ---   isRetrograde 
**
******************************************************/
bool VargaView::isRetrograde( const int &planet )
{
	assert( planet >= 0 && planet < MAX_IOBJECTS_LAGNA_UPA && horoscope );
	return horoscope->isObjectRetrograde( planet );
}







