/*

	This is the source code of

   	M A I T R E Y A
    ===============

	Open source platform for Vedic and western astrology

  File           func.cpp
  Release        4.1
  Author         Martin Pettau
  Copyright (C)  2003-2006 by the author

  Released under the Artistic License as published by the
  Free Software Foundation, read the file 'COPYING' for more information.

*/

#ifdef __GNUG__
	#pragma implementation "func.h"
#endif

#include "func.h"

#include <wx/filename.h>
#include <wx/log.h>
#include <wx/string.h>
#include <wx/tokenzr.h>
#include <wx/utils.h>

#include <locale.h>
#include <math.h>

#include "constants.h"
#include "Conf.h"
#include "Lang.h"

extern Config *config;

/**************************************************************
***
** printNl  --  print a new line to stdout
***
***************************************************************/
void printNl()
{
	printf( "\n" );
}

/**************************************************************
***
** printStr  -- print a string to stdout
***
***************************************************************/
void printStr( const char *s )
{
	printf( "%s\n", s );
}

/**************************************************************
***
** printErr -- print error to stderr
***
***************************************************************/
void printErr( const char *s )
{
	fprintf( stderr, "%s\n", s );
}

/**************************************************************
***
** printErrAndExit -- print Error to stderr and exit
***
***************************************************************/
void printErrAndExit( const char *s, const int code )
{
	fprintf( stderr, "Fatal Error: %s\n", s );
	exit(1);
}

/**************************************************************
***
** printEl -- print an empty line to stdout
***
***************************************************************/
void printEl()
{
	printf( "-----------------------------------------------------------\n" );
}

/*****************************************************
**
**   getRgbValues
**
******************************************************/
void getRgbValues( const int &color, unsigned char &red, unsigned char &green, unsigned char &blue )
{
	int i; 
	blue = (unsigned char)( color % 256 );
	i = ( color - blue ) / 256; 
	green = (unsigned char)(i % 256);    
	red = (unsigned char)(( i - green ) / 256 );
}

/*****************************************************
**
**   getRgbString
**
******************************************************/
void getRgbString( wxChar *buf, const int &color )
{
	unsigned char red, green, blue;

	getRgbValues( color, red, green, blue );
	wxSprintf( buf, wxT( "#%02x%02x%02x" ), red, green, blue );
}

/*****************************************************
**
**   locateFile
**
******************************************************/
wxString locateFile( const wxString &file )
{
	wxString path, fn;

	wxGetEnv( wxT( "PATH" ), &path );
	wxStringTokenizer tk( path, wxT( ":" ));
	while( tk.HasMoreTokens() )
	{
		fn = tk.GetNextToken();
		if ( fn.Last() != wxFileName::GetPathSeparator()) fn << wxFileName::GetPathSeparator();
		fn << file;
		if ( wxFileName::FileExists( fn )) return fn;
	}
	return wxEmptyString;
}


/**************************************************************
***
** 
***
***************************************************************/
double getYearLength( const bool vedic )
{
	int yl = ( vedic ? config->iYearLength : config->hsYearLength );
	double custom_yl = ( vedic ? config->iCustomYearLength : config->hsCustomYearLength );

	switch( yl )
  {
    case 0:
      return 365.25;
    break;
    case 1:
      return 365.2422;
    break;
    case 2:
      return 365;
    break;
    case 3:
      return 360;
    break;
    case 4:
      return custom_yl;
    break;
    default:
			wxLogError( wxT( "Warning: Wrong year length index %d taking default 365.25" ), yl );
      return 365.25;
    break;
  }
}

/**************************************************************
***
**  prints a double into a string and cuts trailing zeros (leaving at least one)
***
***************************************************************/
const wxString printfDouble( const double &d )
{
	wxString s;
	s.Printf( wxT( "%.8f" ), d );
	int i = s.Len();
	wxChar dp = (localeconv()->decimal_point)[0];
	int c = s.Find( dp );
	if ( c > -1 )
	{
		while( i > c+2 )
		{
			if ( s.GetChar( i-1 ) == '0' ) { s = s.Left( i-1 ); }
			else break;
			i--;
		}
	}
	return s;
}

/*****************************************************
**
**   calcYearLength
**
******************************************************/
double calcYearLength( const int sel, const bool vedic )
{
  double custom_yl = ( vedic ? config->iCustomYearLength : config->hsCustomYearLength );
  
  switch( sel )
  {
    case 0:
      return 365.25;
    break;
    case 1:
      return 365.2422;
    break;
    case 2:
      return 365;
    break;
    case 3:
      return 360;
    break;
    case 4:
      return custom_yl;
    break;
    default:
      printf( "Warning: Wrong year length index %d taking default 365.25", sel );
      return 365.25;
    break;
  }
}

/**************************************************************
***
** 
***
***************************************************************/
double getNavamsaPortion( const double &len, const int &pindex, const bool &retro )
{
	double navamsavastha = 0;
	double nav_portion;
	int start_nav;

	assert(( pindex <= ISATURN ) && ( pindex >= 0 ));

	if ( ! retro )
	{
		nav_portion = a_red( len, 30 ) / 3.3333333333;
		start_nav = red12( getRasi( len ) * 9 );
	}
	else
	{
		nav_portion = ( 30 - a_red( len, 30 )) / 3.3333333333;
		start_nav = red12( getRasi( len ) * 9 + 8 );
	}

	int inav_portion = (int)nav_portion;
	int nav_rasi;
	double portion;
	navamsavastha = 0;

	for( int i = 0; i <= inav_portion; i++ )
	{
		if ( i < inav_portion ) portion = 1;
		else portion = nav_portion - (int)nav_portion;

		if ( ! retro )
		{
			nav_rasi = red12( start_nav + i );
		}
		else
		{
			nav_rasi = red12( start_nav - i );
		}

		if ( hasExaltationRasi( pindex, nav_rasi ))
		{
			navamsavastha += 2 * portion;
		}
		else if ( getLord( nav_rasi ) == pindex )
		{
			navamsavastha += 2 * portion;
		}
		else if ( hasDebilationRasi( pindex, nav_rasi ))
		{
			navamsavastha -= portion;
		}
		else if ( isFriend( pindex, getLord( nav_rasi )) == ENEMY )
		{
			navamsavastha -= portion;
		}
		else
		{
			navamsavastha += portion;
		}
	}
	if ( navamsavastha  < 0 ) navamsavastha = 0;
	return navamsavastha;
}


/**************************************************************
***
 * Returns the Greenwich mean time for a given julian date.
 * the return value is between 0 and 24. There's no formating like [hhmmss], etc.
***
***************************************************************/
double getTimeFromJD( const double &jd )
{
	double jjd = jd - .5;
	return ( jjd - (double)(int)jjd ) * 24;
}

/**************************************************************
***
**  Reduces a given double value x modulo the double a (should be positive).
**  The return value is between 0 and a.
***
***************************************************************/
double a_red( const double &x, const double &a )
{
	assert( a > 0 );
	return( x - floor( x / a ) * a );
}


/**************************************************************
***
**   Reduces a given double value modulo 360.
**   The return value is between 0 and 360.
***
***************************************************************/
double red_deg( const double &input )
{
  return a_red( input, 360 );
}



/**************************************************************
***
**  Reduces a given integer modulo 12.
**  The return value is between 0 and 11.
***
***************************************************************/
int red12( const int &input )
{
	int ret = input %12;
	return ret >= 0 ? ret : ret + 12;
}

/**************************************************************
***
** 
***
***************************************************************/
int a_rund( const double &v )
{
	return v >= 0 ? (int)(v+.5) : (int)(v-.5);
}

/**************************************************************
***
** 
***
***************************************************************/
double myatof( const wxString &s )
{
	assert( s );
	wxString err;
	wxChar tmp[256];
	wxStrcpy( tmp, s );
	wxChar *c = tmp;
	wxChar dp = (localeconv()->decimal_point)[0];

	while( *c )
	{
		if ( *c == '.' && *c != dp ) *c = ',';
		if ( *c == ',' && *c != dp ) *c = '.';
		c++;
	}
	wxString s1 = tmp;
	double a = 0;
	if ( ! s1.ToDouble( &a ) && s != wxEmptyString )
	{
		err.Printf( wxT( "Can't convert string to double, string was \"%s\", local decimal point \"%c\", converted string \"%s\", double is %f\n" ), s.c_str(), dp, c, a );
		Cout( err );
	}
	return a;
}

/**************************************************************
***
** 
***
***************************************************************/
void getDasaDurationInts( int &year, int &month, int &day, double length )
{
	double yeard, monthd, dayd;

	yeard = length/getYearLength( true );
	year = (int)yeard;
	monthd = ( yeard - (double)year ) * 12;
	month = (int)monthd;
	dayd = ( monthd - (double)month ) * 30;
	dayd += .00001;
	day = (int)dayd;
}

/**************************************************************
***
** 
***
***************************************************************/
void getDasaDuration( wxString &buf, double length )
{
// YEAR_LEN is 365.25;
#define MY_NULL .00000001
	int years, months, days, hours, minutes, seconds;
	double yeard, monthd, dayd, hourd, minuted, secondd;

	buf = wxT( " --- " );
	wxChar yd, md = 'M', dd;
	yd = ( config->lang == wxT( "de" ) ? 'J' : 'Y' );
	dd = ( config->lang == wxT( "de" ) ? 'T' : 'D' );

	yeard = length/getYearLength( true );
	years = (int)yeard;
	monthd = ( yeard - (double)years ) * 12;
	if ( ::fabs( monthd < MY_NULL ))
	{
		buf.Printf( wxT( "%d%c" ), years, yd );
		//if ( years > 1 ) strcat( buf, "s" );
		return;
	}
	months = (int)monthd;
	dayd = ( monthd - (double)months ) * 30;
	dayd += .00001;
	if ( ::fabs( dayd < MY_NULL ))
	{
		buf.Printf( wxT( "%d%c %d%c" ), years, yd, months, md );
		return;
	}
	days = (int)dayd;
	hourd = ( dayd - (double)days ) * 24;
	if ( years > 0 || months > 0 )
	{
		buf.Printf( wxT( "%d%c %d%c %d%c" ), years, yd, months, md, days, dd );
		return;
	}
	hours = (int)hourd;
	minuted = ( hourd - (double)hours ) * 60;
	minutes = (int)minuted;
	secondd = ( minuted - (double)minutes ) * 60;
	seconds = (int)secondd;
	buf.Printf( wxT( "%d%c %02d:%02d:%02d" ), days, dd, hours, minutes, seconds );
}

/**************************************************************
***
** 
***
***************************************************************/
double getOpposition( const double &v )
{
	return red_deg( v + 180.0 );
}

/**************************************************************
***
** 
***
***************************************************************/
int getRasi( const double &len )
{
	return( (int)( red_deg( len ) / 30 )  );
}

#define USHADHA_BEGIN 266.6666666667
#define ABHIJIT_BEGIN 276.6666666667
#define SRAVANA_BEGIN 280.8888888889
#define DHANISHTA_BEGIN 293.6666666667

#define USHADHA_LEN 10
#define ABHIJIT_LEN 4.2222222222 // = 4:13:20;
#define SRAVANA_LEN 12.444444444 // = 12:26:40
/**************************************************************
***
** 
***
***************************************************************/
double getCorrectedNakshatraLength( const double &len )
{
	// incl. P. Shadha
	if ( len <= USHADHA_BEGIN ) return len;

	// U. Shadha
	else if (( len > USHADHA_BEGIN ) && ( len <= ABHIJIT_BEGIN  ))
		return USHADHA * NAKSHATRA_LEN + NAKSHATRA_LEN / USHADHA_LEN * ( len - USHADHA_BEGIN );

	// Abhijit: upto 280:53:20
	else if (( len > ABHIJIT_BEGIN ) && ( len <= SRAVANA_BEGIN  ))
		return ABHIJIT * NAKSHATRA_LEN + NAKSHATRA_LEN / ABHIJIT_LEN * ( len - ABHIJIT_BEGIN );

	// Sravana
	else if (( len > SRAVANA_BEGIN ) && ( len <= DHANISHTA_BEGIN  ))
		return SRAVANA * NAKSHATRA_LEN + 1.071428571 * ( len - SRAVANA_BEGIN );

	// rest add one nakshatra
	else return len + NAKSHATRA_LEN;
}
/**************************************************************
***
** 
***
***************************************************************/
int getNakshatra( const double &len, const int nsys )
{
	if ( nsys == N27 ) return( (int)a_red( red_deg( len ) / NAKSHATRA_LEN, 27 ));
	else return( (int)( getCorrectedNakshatraLength( len ) / NAKSHATRA_LEN ));
}

/**************************************************************
***
** 
***
***************************************************************/
double getNakshatraLength( const double &len, const int nsys )
{
	if ( nsys == N27 ) return a_red( len, NAKSHATRA_LEN );
	else return a_red( getCorrectedNakshatraLength( len ), NAKSHATRA_LEN );
}

/**************************************************************
***
** 
***
***************************************************************/
/*
double getVedhaNakshatraLength( const double &len )
{

	assert( len >= 0 && len < 360 );
	double clen = getCorrectedNakshatraLength( len );
	int nak = (int)(clen / NAKSHATRA_LEN );
	int vedha_nak = getVedhaNakshatra( nak );
	double nlen = clen - nak * NAKSHATRA_LEN;
	double vedha_len = ( NAKSHATRA_LEN - nlen ) + vedha_nak * NAKSHATRA_LEN;
	double nak_len;

	if ( vedha_nak <= PSHADHA ) return vedha_len;
	else if ( vedha_nak == USHADHA )
	{
		nak_len = vedha_len - USHADHA * NAKSHATRA_LEN;
		assert( nak_len >= 0 && nak_len <= NAKSHATRA_LEN );
		return USHADHA_BEGIN + nak_len * .75;
	}
	else if ( vedha_nak == ABHIJIT )
	{
		nak_len = vedha_len - ABHIJIT * NAKSHATRA_LEN;
		assert( nak_len >= 0 && nak_len <= NAKSHATRA_LEN );
		return ABHIJIT_BEGIN + nak_len * ABHIJIT_LEN / NAKSHATRA_LEN;
	}
	else if ( vedha_nak == SRAVANA )
	{
		nak_len = vedha_len - SRAVANA * NAKSHATRA_LEN;
		assert( nak_len >= 0 && nak_len <= NAKSHATRA_LEN );
		return SRAVANA_BEGIN + nak_len * SRAVANA_LEN / NAKSHATRA_LEN;
	}
	else if ( vedha_nak >= DHANISHTA ) return vedha_len - NAKSHATRA_LEN;
	else assert( 0 );
}
*/

/**************************************************************
***
** 
***
***************************************************************/
/*
int getVedhaNakshatra( const int &nak )
{
	assert( nak >= 0 && nak < 28 );
	switch( nak )
	{
		case ASWINI: return PPHALGUNI; break;
		case BHARANI: return MAKHA; break;
		case KRITTIKA: return SRAVANA; break;
		case ROHINI: return ABHIJIT; break;
		case MRIGASIRA: return USHADHA; break;
		case ARUDRA: return PSHADHA; break;
		case PUNARVASU: return MULA; break;
		case PUSHYAMI: return JYESHTA; break;
		case ASLESHA: return ANURADHA; break;

		case MAKHA: return BHARANI; break;
		case PPHALGUNI: return ASWINI; break;
		case UPHALGUNI: return REVATI; break;
		case HASTHA: return UBHADRA; break;
		case CHITTA: return PBHADRA; break;
		case SWATI: return SATHABISHAK; break;
		case VISAKHA: return DHANISHTA; break;
		case ANURADHA: return ASLESHA; break;
		case JYESHTA: return PUSHYAMI; break;

		case MULA: return PUNARVASU; break;
		case PSHADHA: return ARUDRA; break;
		case USHADHA: return MRIGASIRA; break;
		case ABHIJIT: return ROHINI; break;
		case SRAVANA: return KRITTIKA; break;
		case DHANISHTA: return VISAKHA; break;
		case SATHABISHAK: return SWATI; break;
		case PBHADRA: return CHITTA; break;
		case UBHADRA: return HASTHA; break;
		case REVATI: return UPHALGUNI; break;
		default:
			assert( 0 );
		break;
	}
	assert( 0 );
}
*/
/**************************************************************
***
** 
***
***************************************************************/
int getPada( const double &len )
{
	return( (int)a_red( red_deg( len ) * 4 / NAKSHATRA_LEN, 108 ));
}

/**************************************************************
***
** 
***
***************************************************************/
int isOddFootedRasi( const int &rasi )
{
	assert(( rasi >= 0 ) && ( rasi < 12 ));

	if (( rasi == ARIES )
		|| ( rasi == TAURUS )
		|| ( rasi == GEMINI )
		|| ( rasi == LIBRA )
		|| ( rasi == SCORPIO )
		|| ( rasi == SAGITTARIUS )
	)
		return true;

	return false;
}

/**************************************************************
***
**   getVarna
***
***************************************************************/
int getVarna( const double &len )
{
	assert( len >= 0 && len <= 360 );
	switch ( getRasi( len ) % 4 )
	{
		case 0:
			return 2; // Kshattriya
		break;
		case 1:
			return 1; // Vaishya
		break;
		case 2:
			return 0; // Shudra
		break;
		case 3:
			return 3; // Brahmin
		break;
		default:
			assert( 0 );
			return -1;
		break;
	}
	return -1;
}

/**************************************************************
***
** 
***
***************************************************************/
bool hasJaiminiAspect( const int &pos1, const int &pos2 )
{
	int diff = red12( pos2 - pos1 );

	// CHARA
	if ( ! ( pos1 % 3 ))
	{
		if (( diff == 4 ) || ( diff == 7 ) || ( diff == 10 )) return true;
	}
	// STHIRA
	else if ( ( pos1 % 3 ) == 1 )
	{
		if (( diff == 2 ) || ( diff == 5 ) || ( diff == 8 )) return true;
	}
	// DWISWA
	else
	{
		if (( diff == 3 ) || ( diff == 3 ) || ( diff == 9 )) return true;
	}

	return false;
}


/**************************************************************
***
** 
***
***************************************************************/
int isOddRasi( const double &len )
{
	return( getRasi( len + 30 ) % 2 );
}

/**************************************************************
***
** 
***
***************************************************************/
int isEvenRasi( const double &len )
{
	return( getRasi( len ) % 2 );
}

/**************************************************************
***
** isSampadaRasi
***
***************************************************************/
bool isSampadaRasi( const int &rasi )
{
	if ( rasi == CANCER ) return true;
	if ( rasi == LEO ) return true;
	if ( rasi == VIRGO ) return true;
	if ( rasi == CAPRICORNUS ) return true;
	if ( rasi == AQUARIUS ) return true;
	if ( rasi == PISCES ) return true;
	return false;
}

/**************************************************************
***
** isVishamapadaRasi
***
***************************************************************/
bool isVishamapadaRasi( const int &rasi )
{
	if ( rasi == ARIES ) return true;
	if ( rasi == TAURUS ) return true;
	if ( rasi == GEMINI ) return true;
	if ( rasi == LIBRA ) return true;
	if ( rasi == SCORPIO ) return true;
	if ( rasi == SAGITTARIUS ) return true;
	return false;
}

/**************************************************************
***
** 
***
***************************************************************/
double getRasiLen( const double &len, const bool &reverse )
{
	if ( ! reverse )	return( a_red( len, 30 ));
	else return( 30 - a_red( len, 30 ));
}

/**************************************************************
***
** 
***
***************************************************************/
int between( const double &len, const double &min, const double &max )
{
	return( ( len >= min ) && ( len <= max )  );
}

/**************************************************************
***
** 
***
***************************************************************/
double planetDistance( const double &len1, const double &len2 )
{
	double d = red_deg( fabs( len2 - len1 ));
	if ( d > 180 ) return( 360 - d );
	return d;
}

/**************************************************************
***
** 
***
***************************************************************/
int inAscendant( const double &len, const double &asc )
{
	return( a_red( getRasi( len ) - getRasi( asc ), 12 ) == 0 );
	
	return 0;
}

/**************************************************************
***
** 
***
***************************************************************/
int inTrikona( const double &len, const double &asc )
{
	int b = (int)a_red( getRasi( len ) - getRasi( asc ), 12 );

	if (( b == HOUSE5 ) || ( b == HOUSE9 )) return 1;
	else return 0;
}

/**************************************************************
***
** 
***
***************************************************************/
int inKendra( const double &len, const double &asc )
{

	int b = (int)a_red( getRasi( len ) - getRasi( asc ), 12 );

	if (( b == HOUSE4 ) || ( b == HOUSE7 ) || ( b == HOUSE10 )) return 1;
	else return 0;
}

/**************************************************************
***
** 
***
***************************************************************/
int inMovableSign( const double &len ) 
{
	return( getRasi( len ) % 3 ) == 0;
}

/**************************************************************
***
** 
***
***************************************************************/
int inFixedSign( const double &len )
{
	return( getRasi( len ) % 3 ) == 1;
}

/**************************************************************
***
** 
***
***************************************************************/
int inDualSign( const double &len )
{
	return( getRasi( len ) % 3 ) == 2;
}

/**************************************************************
***
** getAvasthaName
***
***************************************************************/
const wxChar *getAvasthaName( const int &i , const int &avastha )
{
	const wxChar *sayanadi[12] = { wxT( "Sayana" ), wxT( "Upavesana" ), wxT( "Netrapani" ),
	wxT( "Prakasana" ), wxT( "Gamana" ), wxT( "Aagamana" ), wxT( "Sabha" ), wxT( "Agama" ),
	wxT( "Bhojama" ), wxT( "Nrityalipsa" ), wxT( "Kautuka" ), wxT( "Nidra" ) };

if ( i > 1 ) return sayanadi[i - 1];
return sayanadi[11];
}

/**************************************************************
***
** 
***
***************************************************************/
int getLord( const int &sign )
{
	assert(( sign >= ARIES ) && ( sign < 12 ));

	const int k_lord[12] = {
		IMARS, IVENUS, IMERCURY, IMOON, ISUN, IMERCURY,
		IVENUS, IMARS, IJUPITER, ISATURN, ISATURN, IJUPITER
	};
	return k_lord[sign];
}

/**************************************************************
***
** 
***
***************************************************************/
int isFriend( const int &planet1, const int &planet2 )
{
	assert(( planet1 >= ISUN ) && ( planet1 <= IKETU ));
	assert(( planet2 >= ISUN ) && ( planet2 <= IKETU ));

	const int k_friend[9][9] = {
		{FRIEND, FRIEND, FRIEND, NEUTRAL, FRIEND, ENEMY, ENEMY, ENEMY, ENEMY},
		{FRIEND, FRIEND, NEUTRAL, FRIEND, NEUTRAL, NEUTRAL, NEUTRAL, ENEMY, ENEMY},
		{FRIEND, FRIEND, FRIEND, ENEMY, FRIEND, NEUTRAL, NEUTRAL, ENEMY, FRIEND},
		{FRIEND, ENEMY, NEUTRAL, FRIEND, NEUTRAL, FRIEND, NEUTRAL, NEUTRAL, NEUTRAL},
		{FRIEND, FRIEND, FRIEND, ENEMY, FRIEND, ENEMY, NEUTRAL, FRIEND, NEUTRAL},
		{ENEMY, ENEMY, NEUTRAL, FRIEND, NEUTRAL, FRIEND, FRIEND, FRIEND, FRIEND},
		{ENEMY, ENEMY, ENEMY, FRIEND, NEUTRAL, FRIEND, FRIEND, FRIEND, FRIEND},
		{ENEMY, ENEMY, ENEMY, NEUTRAL, FRIEND, FRIEND, FRIEND, FRIEND, FRIEND},
		{ENEMY, ENEMY, FRIEND, NEUTRAL, NEUTRAL, FRIEND, FRIEND, FRIEND, FRIEND}
	};

	return k_friend[planet1][planet2];
}

/**************************************************************
***
** 
***
***************************************************************/
int isRasiPosTempFriend( const int &rasi1, const int &rasi2 )
{
	assert( rasi1 >= 0 && rasi1 < 12 );
	assert( rasi2 >= 0 && rasi2 < 12 );
	int diff = red12( rasi2 - rasi1 );
	if ( diff == 1 || diff == 2 || diff == 3 || diff == 11 || diff == 10 || diff == 9 ) return FRIEND;
	else return ENEMY;
}

/**************************************************************
***
** 
***
***************************************************************/
bool isInMulatrikona( const int &planet, const double &len )
{
	if ( planet == ISUN )
	{
		if (( len >= 120 ) && ( len <= 140 )) return true;
		else return false;
	}
	else if ( planet == IMOON )
	{
		if (( len >= 34 ) && ( len <= 60 )) return true;
		else return false;
	}
	else if ( planet == IMARS )
	{
		if (( len >= 0 ) && ( len <= 12 )) return true;
		else return false;
	}
	else if ( planet == IMERCURY )
	{
		if (( len >= 165 ) && ( len <= 170 )) return true;
		else return false;
	}
	else if ( planet == IJUPITER )
	{
		if (( len >= 240 ) && ( len <= 250 )) return true;
		else return false;
	}
	else if ( planet == IVENUS )
	{
		if (( len >= 180 ) && ( len <= 195 )) return true;
		else return false;
	}
	else if ( planet == ISATURN )
	{
		if (( len >= 300 ) && ( len <= 320 )) return true;
		else return false;
	}
	else if ( planet == IRAHU )
	{
		if (( len >= 60 ) && ( len <= 90 )) return true;
		else return false;
	}
	else if ( planet == IKETU )
	{
		if (( len >= 240 ) && ( len <= 270 )) return true;
		else return false;
	}
	else
	{
		wxLogError( wxT( "ERROR isInMulatrikona planet " ));
		exit(1);
	}
return false;
}

/**************************************************************
***
** 
***
***************************************************************/
int getMulatrikonaRasi( const int &planet )
{
	assert(( planet >= ISUN ) && ( planet <= IKETU ));

	const int k_mulatrikona_sign[9] = { LEO, TAURUS, ARIES, VIRGO, SAGITTARIUS, LIBRA, AQUARIUS, AQUARIUS, SCORPIO };
	//GEMINI, SAGITTARIUS };
	return k_mulatrikona_sign[planet];
}

/**************************************************************
***
** 
***
***************************************************************/
bool hasMulatrikonaRasi( const int &planet, const int &rasi )
{
	assert(( planet >= ISUN ) && ( planet <= IKETU ));
	assert(( rasi >= ARIES ) && ( rasi <= PISCES ));

	return( getMulatrikonaRasi( planet ) == rasi );
}

/**************************************************************
***
** 
***
***************************************************************/
int getExaltationRasi( const int &planet )
{
	assert(( planet >= ISUN ) && ( planet <= IKETU ));

	const int exalt_rasi[9] = { ARIES, TAURUS, CAPRICORNUS, VIRGO, CANCER, PISCES, LIBRA, GEMINI, SAGITTARIUS };
	// TAURUS, SCORPIO };
	return exalt_rasi[planet];
}


/**************************************************************
***
** 
***
***************************************************************/
bool hasExaltationRasi( const int &planet, const int &rasi )
{
	assert(( planet >= ISUN ) && ( planet <= IKETU ));
	assert(( rasi >= ARIES ) && ( rasi <= PISCES ));

	return ( getExaltationRasi( planet ) == rasi );
}

/**************************************************************
***
** 
***
***************************************************************/
bool hasDebilationRasi( const int &planet, const int &rasi )
{
	assert(( planet >= ISUN ) && ( planet <= IKETU ));
	assert(( rasi >= ARIES ) && ( rasi <= PISCES ));

	int erasi = rasi + 6;
	if ( erasi > 11 ) erasi -= 12;
	return ( getExaltationRasi( planet ) == erasi );
}

/**************************************************************
***
** 
***
***************************************************************/
double getAspectValue( const int &planet1, const double &len1, const int &planet2, const double &len2 )
{
	assert( ( planet1 >= ISUN ) && ( planet1 <= ISATURN ));
	assert( ( planet2 >= ISUN ) && ( planet2 <= IKETU ));

	double ret = 0;
	double asplen = red_deg( len2 - len1 );

	if (( asplen >= 30 ) && ( asplen <= 60 ))
	{
		ret = ( asplen - 30 ) * .5;
		if ( planet1 == ISATURN )
		{
			ret = ( asplen - 30 ) * 2;
		}
	}
	else if (( asplen > 60 ) && ( asplen <= 90 ))
	{
		ret = asplen - 45;
		if ( planet1 == ISATURN )
		{
			ret = 45 + ( 90 - asplen ) * .5;
		}
	}
	else if (( asplen > 90 ) && ( asplen <= 120 ))
	{
		ret = ( 120 - asplen ) * .5 + 30;
		if ( planet1 == IMARS )
		{
			ret = 45 + ( asplen - 90 ) * .5;
		}
		if ( planet1 == IJUPITER )
		{
			ret = 45 + ( asplen - 90 ) * .5;
		}
	}
	else if (( asplen > 120 ) && ( asplen <= 150 ))
	{
		ret = 150 - asplen;
		if ( planet1 == IMARS )
		{
			ret = ( 150 - asplen ) * 2;
		}
		if ( planet1 == IJUPITER )
		{
			ret = ( 150 - asplen ) * 2;
		}
	}
	else if (( asplen > 150 ) && ( asplen <= 180 ))
	{
		ret = ( asplen - 150 ) * 2;
	}
	else if (( asplen > 180 ) && ( asplen <= 300 ))
	{
		ret = ( 300 - asplen ) * .5;
		if ( planet1 == IMARS )
		{
			if (( asplen >= 180 ) && ( asplen <=210 ))
			{
				ret = 60;
			}
			if (( asplen >= 210 ) && ( asplen <=240 ))
			{
				ret = 270 - asplen;
			}
		}
		if ( planet1 == IJUPITER )
		{
			if (( asplen >= 210 ) && ( asplen <=240 ))
			{
				ret = 45 + ( asplen - 210 ) * .5;
			}
			else if (( asplen >= 240 ) && ( asplen <=270 ))
			{
				ret = 15 + ( 270 - asplen ) * 1.5;
			}
		}
		if ( planet1 == ISATURN )
		{
			if (( asplen >= 240 ) && ( asplen <=270 ))
			{
				//ret = 30 + asplen - 240;
				ret = asplen - 210;
			}
			else if (( asplen >= 270 ) && ( asplen <=300 ))
			{
				ret = ( 300 - asplen ) * 2;
			}
		}
	}
	assert(( ret >= 0 ) && ( ret <= 60 ));
	return ret;
}

/**************************************************************
***
**   Calculate geometrical midpoint for composite horoscope
***
***************************************************************/
double calcGeometricalMidpoint( const double &len1, const double &len2 )
{
		double dist = red_deg( len2 - len1 );
		return( dist < 180 ? red_deg( len1 + dist/2 ) : red_deg( len2 + ( 360 - dist ) /2 ) );
}

/**************************************************************
***
**   Berechnung von Treffern im N-Gradkreis
***
***************************************************************/
int berechneNGradTreffer( const double &len1, const double &len2, const int &maxorbis, double &orbis, const int gradkreis ) // = GRADKREIS_45 )
{
	assert( maxorbis >= 0 );
	orbis = 100;
	double dmaxorbis = maxorbis;
	dmaxorbis /= 60;

	double dist, adist;

	dist = red_deg( len2 - len1 );
	adist = dist > 180 ? 360 - dist : dist;
	if ( adist < dmaxorbis )
	{
		orbis = dist < 180 ? adist : -adist;
		return TREFFER_360;
	}
	if ( gradkreis == GRADKREIS_360 ) return 0;

	dist = a_red( len2 - len1, 180 );
	adist = dist > 90 ? 180 - dist : dist;
	if ( adist < dmaxorbis )
	{
		orbis = dist < 90 ? adist : -adist;
		return TREFFER_180;
	}
	if ( gradkreis == GRADKREIS_180 ) return 0;

	dist = a_red( len2 - len1, 90 );
	adist = dist > 45 ? 90 - dist : dist;
	if ( adist < dmaxorbis )
	{
		orbis = dist < 45 ? adist : -adist;
		return TREFFER_90;
	}
	if ( gradkreis == GRADKREIS_90 ) return 0;

	dist = a_red( len2 - len1, 45 );
	adist = dist > 22.5 ? 45 - dist : dist;
	if ( adist < dmaxorbis )
	{
		orbis = dist < 22.5 ? adist : -adist;
		return TREFFER_45;
	}
	if ( gradkreis == GRADKREIS_45 ) return 0;

	dist = a_red( len2 - len1, 22.5 );
	adist = dist > 11.25 ? 22.5 - dist : dist;
	if ( adist < dmaxorbis )
	{
		orbis = dist < 11.25 ? adist : -adist;
		return TREFFER_225;
	}
	if ( gradkreis == GRADKREIS_225 ) return 0;

	dist = a_red( len2 - len1, 11.25 );
	adist = dist > 5.625 ? 11.25 - dist : dist;
	if ( adist < dmaxorbis )
	{
		orbis = dist < 5.625 ? adist : -adist;
		return TREFFER_1125;
	}
	if ( gradkreis == GRADKREIS_1125 ) return 0;

	dist = a_red( len2 - len1, 5.625 );
	adist = dist > 2.8125 ? 5.625 - dist : dist;
	if ( adist < dmaxorbis )
	{
		orbis = dist < 2.8125 ? adist : -adist;
		return TREFFER_5625;
	}
	return 0;
}

double getKreisgrad( const int &gradkreis )
{
	if ( gradkreis == GRADKREIS_360 ) return 360;
	else if ( gradkreis == GRADKREIS_180 ) return 180;
	else if ( gradkreis == GRADKREIS_90 ) return 90;
	else if ( gradkreis == GRADKREIS_45 ) return 45;
	else if ( gradkreis == GRADKREIS_225 ) return 22.5;
	else if ( gradkreis == GRADKREIS_1125 ) return 11.25;
	else if ( gradkreis == GRADKREIS_5625 ) return 5.625;
	else assert( false );
	return 0;
}


/**************************************************************************
***
**  Monatslaenge berechnen
***
**************************************************************************/
int calcMonthLength( int monat, int jahr )
{
  if ( monat == 1 || monat == 3 || monat == 5 || monat == 7 || monat == 8 || monat == 10 || monat == 12 ) return 31;
  if ( monat == 4 || monat == 6 || monat == 9 || monat == 11 ) return 30;
  assert( monat == 2 );
  if ( jahr % 4 )  return 28;
  else
  {
    if (( jahr % 100 ) != 0 ) return 29;
    if (( jahr % 100 ) == 0 && ( jahr % 400 ) != 0 ) return 28;
    return 29;
  }
}



